#!/bin/sh
# 
# Magisk init script service.sh for the Magisk Module to create a new dynamic partition
#
# History
#   07.11.2024 /bs
#     initial release
#     
#
__TRUE=0
__FALSE=1

MODPATH="$(dirname $0)"

MODULE_NAME="$( grep "^id=" $MODPATH/module.prop 2>/dev/null  | cut -f2 -d "=" )"
[ "${MODULE_NAME}"x = ""x ] && MODULE_NAME="unknown_magisk_module"

CUR_USER_ID="$( id -un )"


# -----------------------------------------------------------------------------
#
if [[ $0 == *service.sh ]] ; then
  CUR_LOGFILE="/data/local/tmp/process_dynamic_partition_service.log"
  SERVICE_SCRIPT_RUNNING=${__TRUE}
else
  CUR_LOGFILE="/data/local/tmp/process_dynamic_partition.action.log"
  SERVICE_SCRIPT_RUNNING=${__FALSE}
fi

if ! tty -s ; then
#
# always write the messages to the log file
#
  if [ 0 = 0 -o -r /data/local/tmp/debug ] ; then
    exec 1>"${CUR_LOGFILE}" 2>&1
  fi
fi

if [ -r /data/local/tmp/debug ] ; then
  set -x
fi

# -----------------------------------------------------------------------------

SERVICE_SCRIPT_CONFIG_FILE="/data/adb/dynamic_partition_service.conf"

# the script service.sh does nothing if this file exists
#
SEMFILE="/data/local/tmp/IGNORE_PERSISTENT_PARTITION"


MOUNT_POINT_MOUNTED=${__FALSE}

# ---------------------------------------------------------------------

function print_message_prefix {
  printf "[$( date +"%Y-%m-%d %H:%M:%S" )]"
}

function LogMsg {
  LOGMSG_THISMSG=""

  if [ "$1"x = "-"x ] ; then
    shift
    LOGMSG_THISMSG="$@"
  else
    LOGMSG_THISMSG="$( print_message_prefix ) $@"
  fi

  echo "${LOGMSG_THISMSG}"

#  [ "${CUR_LOGFILE}"x != ""x ] &&  echo "${LOGMSG_THISMSG}" >>"${CUR_LOGFILE}"

  return ${__TRUE}
}

function LogInfo {
  LOGINFO_THISMSG=""

  if [[ "$1"x == "-"x ]] ; then
    shift
    LOGINFO_THISMSG="$@" 
  else
    LOGINFO_THISMSG="INFO: $@"
  fi
  
  [[ ${VERBOSE} == ${__TRUE} ]] && LogMsg "${LOGINFO_THISMSG}" >&2 || return ${__FALSE}
}

function LogWarning {
  LogMsg "WARNING: $@" >&2
}

function LogError {
  LogMsg "ERROR: $@" >&2
}

function die {
  DIE_THISRC=$1

  [ "${THISRC}"x = ""x ] && DIE_THISRC=0
  
  [ $# -gt 0 ] && shift

  DIE_THISMSG="$*"
  
  if [ "${DIE_THISRC}"x = "0"x -o "${DIE_THISRC}"x = "1"x ] ; then
     [ "${DIE_THISMSG}"x != ""x ] && LogMsg "${DIE_THISMSG} (RC=${DIE_THISRC})"
  else
    ERRORS_FOUND=${__TRUE}
    LogError "${DIE_THISMSG} (RC=${DIE_THISRC})"
  fi
  exit ${DIE_THISRC}
}
# ---------------------------------------------------------------------

if [ ${SERVICE_SCRIPT_RUNNING} = ${__TRUE} ] ; then
  if [ -r "${SEMFILE}" ] ; then
    LogMsg "The file \"${SEMFILE}\" exists -- NOT mounting the persistent partition"
    exit 0
  fi
fi

# ---------------------------------------------------------------------

if [ "${CUR_USER_ID}"x != "root"x ] ; then
  LogError "The script $0 must be executed by the user root"
  exit 5
fi

# ---------------------------------------------------------------------

function mount_point_mounted {
  THISRC=${__FALSE}
  
  CUR_MOUNT_DEVICE=""
  CUR_BASE_DIR_MOUNT_DEVICE=""
    
  CUR_MOUNT_DEVICE="$(  df -h "${MOUNT_POINT}" 2>/dev/null | tail -1 )"
  CUR_MOUNT_DEVICE="${CUR_MOUNT_DEVICE%% *}"
  
  CUR_BASE_DIR_MOUNT_DEVICE="$(  df -h "${MOUNT_POINT%/*}" 2>/dev/null | tail -1 )"
  CUR_BASE_DIR_MOUNT_DEVICE="${CUR_BASE_DIR_MOUNT_DEVICE%% *}"

  [ "${CUR_MOUNT_DEVICE}"x != "${CUR_BASE_DIR_MOUNT_DEVICE}"x ] && THISRC=${__TRUE}

  return ${THISRC}
}

# ---------------------------------------------------------------------

function umount_dynamic_partition {
  THISRC=${__FALSE}

   LogMsg "Umounting the mount point \"${MOUNT_POINT}\" ..."
   umount "${MOUNT_POINT}"
}

# ---------------------------------------------------------------------

function mount_dynamic_partition {
  THISRC=${__FALSE}

  MOUNT_WAIT_TIME=60
  
  LogMsg "Mounting now \"/dev/block/mapper/${LOGICAL_DEVICE_NAME}\" to \"${MOUNT_POINT}\" ..."
  LogMsg "Waiting up to ${MOUNT_WAIT_TIME} seconds to mount the device"

  i=0
  
  while [ $i -lt ${MOUNT_WAIT_TIME} ] ; do

    mount ${MOUNT_OPTIONS} "/dev/block/mapper/${LOGICAL_DEVICE_NAME}" "${MOUNT_POINT}" && THISRC=${__TRUE}

    if [ ${THISRC} = ${__TRUE} ]; then
      break
    fi

    let i=i+10
    sleep 10

  done
  
  if [ ${THISRC} = ${__TRUE}  ] ; then
    LogMsg "The mount succeeded after $i seconds"
      
    LogMsg "-"
    LogMsg "-" "$( df -h "${MOUNT_POINT}" 2>&1 )"
    LogMsg "-"
  else
    LogError "The mount did not wori -- either the logical device or the mount point does not exist"
  fi 

  return ${THISRC}
}

function correct_mount_point_permissions {

    LogMsg "Changing the owner and group for the mount point \"${MOUNT_POINT}\" to \"${MOUNT_POINT_OWNER}:${MOUNT_POINT_GROUP}\" ..."
    CUR_OUTPUT="$( chown ${MOUNT_POINT_OWNER}:${MOUNT_POINT_GROUP} "${MOUNT_POINT}" 2>&1 )"
    if [ $? -gt 1 ] ; then
      LogMsg "-" "${CUR_OUTPUT}"
      LogError "Error changing the owner and group for the mount point \"${MOUNT_POINT}\" to \"${MOUNT_POINT_OWNER}:${MOUNT_POINT_GROUP}\" "
      THISRC=${__FALSE}
      CONTINUE=${__FALSE}
    fi

    LogMsg "Setting the permissions for the mount point \"${MOUNT_POINT}\" to \"${MOUNT_POINT_PERMISSIONS}\" ..."
    CUR_OUTPUT="$( ${DEBUG_PREFIX} ; chmod ${MOUNT_POINT_PERMISSIONS} "${MOUNT_POINT}" 2>&1 )"
    if [ $? -gt 1 ] ; then
      LogMsg "-" "${CUR_OUTPUT}"
      LogError "Error setting the permissions for the mount point \"${MOUNT_POINT}\" to \"${MOUNT_POINT_PERMISSIONS}\" "
      THISRC=${__FALSE}
      CONTINUE=${__FALSE}
    fi

    LogMsg "Setting the SELinux context for the mount point \"${MOUNT_POINT}\" to \"${MOUNT_POINT_SELINUX_CONTEXT}\" ..."
#    CUR_OUTPUT="$( ${DEBUG_PREFIX} ; /data/adb/magisk/busybox chcon -R ${MOUNT_POINT_SELINUX_CONTEXT} "${MOUNT_POINT}" 2>&1 )"
    CUR_OUTPUT="$( ${DEBUG_PREFIX} ; chcon -R ${MOUNT_POINT_SELINUX_CONTEXT} "${MOUNT_POINT}" 2>&1 )"

    if [ $? -gt 1 ] ; then
      LogMsg "-" "${CUR_OUTPUT}"
      LogError "Error setting the SELinux context for the mount point \"${MOUNT_POINT}\" to \"${MOUNT_POINT_SELINUX_CONTEXT}\" "
      THISRC=${__FALSE}
      CONTINUE=${__FALSE}
    fi

    LogMsg "The permissions for the mount point \"${MOUNT_POINT}\" are now:"
    LogMsg ""
    
    ls -ldZ "${MOUNT_POINT}"
    LogMsg ""
}

# ---------------------------------------------------------------------
# read the config for the persisent data from the config file 
# created by process_dynamic_partition.sh
#
if [ ! -r "${SERVICE_SCRIPT_CONFIG_FILE}" ] ; then
  die 100 "The config file \"${SERVICE_SCRIPT_CONFIG_FILE}\" does not exist"  
else
  LogMsg "Reading the config file \"${SERVICE_SCRIPT_CONFIG_FILE}\" ..."
  . "${SERVICE_SCRIPT_CONFIG_FILE}"
fi


CONTINUE=${__TRUE}

mount_point_mounted 
MOUNT_POINT_MOUNTED=$?

case $0 in 

 *service.sh )
  if [  ${MOUNT_POINT_MOUNTED} = ${__TRUE} ] ; then
    LogMsg "The mount point \"${MOUNT_POINT}\" is already mounted:"
    LogMsg "-"
    LogMsg "-" "$( df -h ${MOUNT_POINT} 2>&1 )"
    LogMsg "-"
    CONTINUE=${__FALSE}
  else
    LogMsg "-" "The mount point \"${MOUNT_POINT}\" is not yet mounted"
  fi
  
  if [ ${CONTINUE} = ${__TRUE} ] ; then  
  
    if [ "${MOUNT_WAIT_TIME}"x != ""x ] ; then
      LogMsg "Waiting now for ${MOUNT_WAIT_TIME} seconds ..."
      sleep ${MOUNT_WAIT_TIME}
    fi
    
    mount_dynamic_partition
    correct_mount_point_permissions
  fi
  ;;

  *action.sh )
    if [ ${MOUNT_POINT_MOUNTED} = ${__TRUE} ] ; then
      umount_dynamic_partition
    else
      mount_dynamic_partition
      correct_mount_point_permissions
    fi
    ;;
esac


# -----


