#!/system/bin/sh
#
#h#
#h# process_dynamic_partition.sh -- create, mount, umount, or delete a dynamic partition
#h#
#h# Usage:
#h#   process_dynamic_partition.sh [-f configfile] [-l logfile] [-v|--verbose] [-k|--keep] [-y|--yes] [-e|--ignore_error] [var=value] [create|delete|mount|umount|status|update|refresh] 
#h#
#h# Parameter:
#h#   -f configfile
#h#     The default config file for this script is /etc/process_dynamic_partition.conf
#h#     use the parameter "-f none" to run without a config file
#h#
#h#   -l logfile
#h#     The default log file used by this script is /sdcard/Download/<scriptname>.log
#h#     use the parameter "-l none" to run without a log file
#h#
#h#   -v
#h#     print verbose messages
#h#
#h#   -k 
#h#     do not delete temporary files at script end
#h#
#h#   -y
#h#     assume "yes" for all user requests
#h#
#h#   -e 
#h#     ignore errors, without this parameter the script stops after an action failed
#h#
#h#   var=value
#h#     set the variable "var" to the value "value" at start of the script 
#h#     (before reading the config file)
#h#
#h#   create 
#h#     create the dynamic partition and the logical device 
#h#
#h#   delete 
#h#     delete the logical device 
#h#     Note: The dynamic partition must be deleted manually in fastboot mode 
#h#           (see the instructions printed by the script)
#h#
#h#   mount 
#h#     create the mount point and mount the dynamic partition
#h#
#h#   umount
#h#     umount the dynamic partition
#h#
#h#   status
#h#     print the status of the dynamic partition, the logical device, and the mount point
#h#
#h#   update 
#h#     update the config file for the service.sh script to mount the new dynamic partition
#h#     The config file for the service.sh script is the file /data/adb/dynamic_partition_service.conf
#h#
#h#   refresh
#h#     set the permissions for the mount point
#h#
#h# The parameters can be used in any order; several action parameters are permitted
#h#
#
# History
#   07.11.2024 /bs
#    initial release
#

# -----------------------------------------------------------------------------

DISCLAIMER="
 ***************************************************************************
 CAUTION:

 This script can damage or destroy the data on the phone.

 Only use this script if you have a complete backup of the data on the phone 
 and know how to reinstall the operating system on the phone from scratch

 ***************************************************************************

"

# -----------------------------------------------------------------------------
# constants

__TRUE=0
__FALSE=1

# -----------------------------------------------------------------------------
# global variables
#

SCRIPT_NAME="${0##*/}"

CUR_SCRIPT_PARAMETER="$*"

CUR_USER="$( id -un )"

# default values for the parameter
#
VERBOSE=${__FALSE}

KEEP_TEMPORARY_FILES=${__FALSE}

ASSUME_YES=${__FALSE}

IGNORE_ERROR=${__FALSE}

# default log file for this script
#
DEFAULT_LOGFILE="/sdcard/Download/process_dynamic_partition.log"


# default config file for this script
#
DEFAULT_CONFIG_FILE="/etc/process_dynamic_partition.conf"


# config file for the service.sh script 
#
SERVICE_SCRIPT_CONFIG_FILE="/data/adb/dynamic_partition_service.conf"


# add the known directories with the required binaries to the PATH if not yet done
#
# /system/bin/ro2rw is the directory used in the MagiskModule with the ro2rw tools
#
for CUR_DIR in /system/usr/create_dynamic_partition_tools /system/bin/ro2rw ; do
  if [ -d "${CUR_DIR}" ] ; then
    [[ ":${PATH}:" != *:${CUR_DIR}:* ]] && PATH="${PATH}:${CUR_DIR}"
  fi
done
export PATH

  
# global variables
#
# do not write the script end message if this variable is ${__TRUE} (this is only an ugly work around...)
#
NO_HEADER=${__FALSE}

# this variable is set to "set -x" if the parameter "-v" is used
# (":" is the NOP command for shell scripts)
#
DEBUG_PREFIX=":"

# list of files that should be deleted at script end if the parameter "-k" is NOT used
#
FILES_TO_DELETE=""

# default values for the variables that can be changed in the config file 
# (use the variable name without the leading "DEFAULT_" in the config file)
#

DEFAULT_TMP_DIR="/sdcard/Download"

DEFAULT_PARTITION_NAME="persistentdata"

# default partition size can be any value accepted by dd for the parameter "bs="
# or the fully qualified name of an existing file or device: in this case the size of the new partition is the size of the file/device
#
DEFAULT_PARTITION_SIZE="100M"

DEFAULT_LOGICAL_DEVICE_NAME="persistentdata"

DEFAULT_MOUNT_POINT="/sdcard/Documents/persistentdata"

DEFAULT_MOUNT_POINT_OWNER="media_rw"
DEFAULT_MOUNT_POINT_GROUP="media_rw"
DEFAULT_MOUNT_POINT_PERMISSIONS="1777"
DEFAULT_MOUNT_POINT_SELINUX_CONTEXT="u:object_r:media_rw_data_file:s0"
# or : u:object_r:tmpfs:s0

DEFAULT_SUPER_PARTITION_NAME="super"

DEFAULT_SUPER_PARTITION="/dev/block/by-name/${DEFAULT_SUPER_PARTITION_NAME}"

DEFAULT_DISK_DEVICE="$( ls -l "${DEFAULT_SUPER_PARTITION}" | awk '{ print $NF }' | tr -d "[0-9]" )"

# maximum time to wait that the umount of the new device is successfull
#
DEFAULT_UMOUNT_WAIT_TIME=10

# time in seconds to wait before creating the filesystem on the new device
#
DEFAULT_MKFS_WAIT_TIME=5

# time in seconds to wait before the filesystem is mounted
#
DEFAULT_MOUNT_WAIT_TIME=5

# time in seconds to wait after creating the partition before creating the logical volume
# 
DEFAULT_DMCTL_CREATE_WAIT_TIME=10

# required executables
#
DEFAULT_GETPROP="$( which getprop 2>/dev/null )"
DEFAULT_LPDUMP="$( which lpdump 2>/dev/null )"
DEFAULT_DMCTL="$( which dmctl 2>/dev/null )"
DEFAULT_MOUNT="$( which mount 2>/dev/null )"
DEFAULT_UMOUNT="$( which umount 2>/dev/null )"

# /system/usr/create_dynamic_partition_tools/dd is a symbolic link to the busybox from Magisk
# in some Android version the standard dd command does not support parameter like "bs=100MB"
#DEFAULT_DD="$( which dd 2>/dev/null )"
#
MAGISK_DD="/system/usr/create_dynamic_partition_tools/dd" 
[ ! -x "${MAGISK_DD}" ] && MAGISK_DD=""
DEFAULT_DD="${MAGISK_DD:=$( which dd 2>/dev/null )}"

DEFAULT_DF="$( which df 2>/dev/null )"
DEFAULT_LPADD="$( which lpadd 2>/dev/null )"
DEFAULT_MKFS="$( which mkfs.ext3  2>/dev/null )"
DEFAULT_AWK="$( which awk 2>/dev/null )"

# optional executables
#
DEFAULT_PARTED="$( which parted 2>/dev/null )"

# ---------------------------------------------------------------------

# global variables that should not be used in the config file
#
REQUIRED_EXECUTABLES="GETPROP LPDUMP DMCTL MOUNT UMOUNT DD DF AWK MKFS"

OPTIONAL_EXECUTABLES="PARTED"

EXECUTABLES="${REQUIRED_EXECUTABLES} ${OPTIONAL_EXECUTABLES}"

# ---------------------------------------------------------------------
# signal handler for the house keeping
#
function cleanup {
  typeset THISRC=0

  typeset TMPFILE=""
  
  LogInfo "Housekeeping started ..."
  
  for TMPFILE in ${FILES_TO_DELETE} ; do
    if [ -r "${TMPFILE}" ] ; then
      if [ ${KEEP_TEMPORARY_FILES} = ${__TRUE} ] ; then
        LogMsg "Keeping the temporary file \"${TMPFILE}\" "
      else
        LogInfo "Deleting the temporary file \"${TMPFILE}\" ..."
        \rm -f "${TMPFILE}"
      fi
    fi
  done
  
  [ ${NO_HEADER} = ${__FALSE} ] &&  LogMsg "${SCRIPT_NAME} ended"
  
  return ${THISRC}
}

# ---------------------------------------------------------------------
# install a trap handler for house keeping
#
trap "cleanup"  0

# ---------------------------------------------------------------------


function print_message_prefix {
  printf "[$( date +"%Y-%m-%d %H:%M:%S" )]"
}

function LogMsg {
  typeset THISMSG=""

  if [ "$1"x = "-"x ] ; then
    shift
    THISMSG="$@"
  else
    THISMSG="$( print_message_prefix ) $@"
  fi

  echo "${THISMSG}"

  [ "${CUR_LOGFILE}"x != ""x ] &&  echo "${THISMSG}" >>"${CUR_LOGFILE}"

  return ${__TRUE}
}

function LogInfo {
  typeset THISMSG=""

  if [[ "$1"x == "-"x ]] ; then
    shift
    THISMSG="$@" 
  else
    THISMSG="INFO: $@"
  fi
  
  [[ ${VERBOSE} == ${__TRUE} ]] && LogMsg "${THISMSG}" >&2 || return ${__FALSE}
}

function LogWarning {
  LogMsg "WARNING: $@" >&2
}

function LogError {
  LogMsg "ERROR: $@" >&2
}

function die {
  typeset THISRC=$1

  [ "${THISRC}"x = ""x ] && THISRC=0
  
  [ $# -gt 0 ] && shift

  typeset THISMSG="$*"
  
  if [ "${THISRC}"x = "0"x -o "${THISRC}"x = "1"x ] ; then
     [ "${THISMSG}"x != ""x ] && LogMsg "${THISMSG} (RC=${THISRC})"
  else
    ERRORS_FOUND=${__TRUE}
    LogError "${THISMSG} (RC=${THISRC})"
  fi
  exit ${THISRC}
}

# ---------------------------------------------------------------------
# retrieve the current status of the dynamic partition, the logical device, and the mount point
#
# Usage : get_current_status [partition_config] [logical_device_config]  [mount_device_config] [all]
#
# Default if no parameter is used is "all"
#
function get_current_status {
  typeset THISRC=${__TRUE}

  typeset CONFIG_VARIABLES="$*"
  typeset CUR_CONFIG_VAR=""

# default: get the status of all config data
#  
  CONFIG_VARIABLES="${CONFIG_VARIABLES:=all}"
  
#
# get the current status
#
  for CUR_CONFIG_VAR in ${CONFIG_VARIABLES} ; do
    
    case ${CUR_CONFIG_VAR} in

      partition_config | partition | all )
        CUR_PARTITION_CONFIG="$( ${LPDUMP} -s ${CURRENT_SLOT} | grep "super:"  | grep " ${PARTITION_NAME} " )"
        ;;
    esac
    
    case ${CUR_CONFIG_VAR} in
      logical_device_config | all ) 
        CUR_LOGICAL_DEVICE_CONFIG="$( ${DMCTL} list devices | grep "^${LOGICAL_DEVICE_NAME} " )"
        ;;
    esac
    
    case ${CUR_CONFIG_VAR} in

      logical_device | logical_device_config | all ) 
        CUR_LOGICAL_DEVICE="$(  ${DMCTL} getpath "${LOGICAL_DEVICE_NAME}" 2>/dev/null )"
        ;;
    esac
    
    case ${CUR_CONFIG_VAR} in

      mount_device_config | mount_device | all ) 
        CUR_MOUNT_DEVICE="$( df -h "${MOUNT_POINT}" 2>/dev/null | tail -1  | awk '{ print $1 }' )"
        CUR_BASE_DIR_MOUNT_DEVICE="$( df -h "${MOUNT_POINT%/*}" 2>/dev/null | tail -1 | awk '{ print $1 }' )"
 
        if [ "${CUR_MOUNT_DEVICE}"x != "${CUR_BASE_DIR_MOUNT_DEVICE}"x ] ; then
          CUR_MOUNT_CONFIG="$(  ${MOUNT} | grep " ${REAL_MOUNT_POINT} " )"
          CUR_MOUNT_POINT_PERMISSIONS="$(  ls -ldZ "${MOUNT_POINT}" 2>/dev/null ; echo ; ${PREFIX} df -h "${MOUNT_POINT}" 2>/dev/null )"
        else
          CUR_MOUNT_CONFIG=""
          CUR_MOUNT_POINT_PERMISSIONS=""
        fi
        ;;
      
    esac
  done
  
  return ${THISRC}
}

# ---------------------------------------------------------------------
# create the dynamic partition and the logical device for the partition
#
function create_dynamic_partition {
  typeset THISRC=${__TRUE}
  typeset CONTINUE=${__TRUE}
 
  typeset CUR_OUTPUT=""

  typeset CUR_LOGCIAL_START_SECTOR=0
  typeset CUR_PARTITION_START_SECTOR=0
  typeset CUR_PARTITION_END_SECTOR=0
  typeset CUR_PARTITION_SIZE=0
  typeset CUR_LINE=""
  typeset DMCTL_COMMANDS=""

  typeset TMPFILE0="${TMP_DIR}/new_partition_size_template.$$.img"
  typeset TMPFILE1="${TMP_DIR}/new_partition_layout.$$"
  typeset TMPFILE2="${TMP_DIR}/dmctl_input.$$"

# the filesystem is only created after creating the new logical device
# ToDo: Add a check for a filesystem on the device
#
  typeset CREATE_FS=${__FALSE}
  
# update the global variable with the list of files to delete at script end
#
  FILES_TO_DELETE="${FILES_TO_DELETE} ${TMPFILE0} ${TMPFILE1} ${TMPFILE2}"
      
  LogMsg "Creating the dynamic partition ..."

# lpadd usage:
#
# lpadd /dev/block/by-name/super <partition_name> default /dev/block/by-name/boot_a 
#
  if [ ${CONTINUE} = ${__TRUE} ] ; then
    if [ "${CUR_PARTITION_CONFIG}"x != ""x ] ; then
      LogMsg "The partition already exists:"
      LogMsg "-" "${CUR_PARTITION_CONFIG}"
      THISRC=${__TRUE}
      CONTINUE=${__FALSE}
    else
     
      if [[ ${PARTITION_SIZE} == /* ]] ; then
        if [ ! -r "${PARTITION_SIZE}" ] ;then
          LogError "The template for the size of the new partition, \"${PARTITION_SIZE}\", does not exist"
          CONTINUE=${__FALSE}
          THISRC=${__FALSE}
        else        
          SIZE_TEMPLATE="${PARTITION_SIZE}"

          LogMsg "Using the file/device \"${SIZE_TEMPLATE}\" as template for the size of the new partition"
        fi
      else
#
# lpadd does not support a parameter for the size of the partition - the size of the new partition is defined
# via an existing device or file 
#     
        LogMsg "Creating the temporary file \"${TMPFILE0}\" ..."
        CUR_OUTPUT="$( ${DEBUG_PREFIX} ; ${DD} if=/dev/zero of=${TMPFILE0} bs=${PARTITION_SIZE} count=1 2>&1 )"
        if [ $? -ne 0 ] ; then
          LogMsg "-" "${CUR_OUTPUT}"
          LogError "Error creating the temporary file \"${TMPFILE0}\" "
          CONTINUE=${__FALSE}
          THISRC=${__FALSE}
        else
          SIZE_TEMPLATE="${TMPFILE0}"
        fi           
      fi
    fi
    
    if [ ${CONTINUE} = ${__TRUE} ] ; then
      LogMsg "Creating the dynamic partition \"${PARTITION_NAME}\" ..."

      CUR_OUTPUT="$( ${DEBUG_PREFIX} ; ${LPADD} "${SUPER_PARTITION}"   "${PARTITION_NAME}" default "${SIZE_TEMPLATE}" 2>&1 )"
      if [ $? -ne 0 ] ; then
        LogMsg "-" "${CUR_OUTPUT}"
        LogError "Error creating the dynamic partition \"${PARTITION_NAME}\" "
        CONTINUE=${__FALSE}
        THISRC=${__FALSE}
      else
#
# re-read the partition config
#      
        get_current_status partition_config
      fi         
    fi
    
    [ ${THISRC} = ${__TRUE} ] && CONTINUE=${__TRUE}

  fi

# dmctl usage:
#  
# dmctl create [LOGICAL_DEVICE_NAME] linear [LOGICAL_PARTITION_START] [SIZE_OF_THIS_PART_OF_THE_PARTITION] /dev/block/by-name/super [PARTITION_START_SECTOR]
#
# Logical devices can consist of more than one partition
# 
# For unfragmented logical devices LOGICAL_PARTITION_START is 0 and SIZE_OF_THIS_PART_OF_THE_PARTITION is the length of the partition
#
# example:
#
# dmctl create system_a_new linear 0 7438680 /dev/block/by-name/super 3411968 
#
  if [ ${CONTINUE} = ${__TRUE} ] ; then
    if [ "${CUR_LOGICAL_DEVICE}"x != ""x ] ; then
      LogMsg "The logical device already exists:"
      LogMsg "-" "${CUR_LOGICAL_DEVICE_CONFIG}"
    else
      LogMsg "Creating the logical device \"${LOGICAL_DEVICE_NAME}\" ..."
#
# dmctl only supports fragmented logical devices if the input for the command is in a file ...
#      
      echo "${CUR_PARTITION_CONFIG}"  >"${TMPFILE1}"
  
      DMCTL_COMMANDS="create ${LOGICAL_DEVICE_NAME}"
 
      CUR_LOGICAL_START_SECTOR="0"
  
      while read CUR_LINE ; do

        CUR_PARTITION_START_SECTOR="$( echo "${CUR_LINE}" | awk '{ print $2 }'   )"
        CUR_PARTITION_END_SECTOR="$(   echo "${CUR_LINE}" | awk '{ print $4 }'   )"
        CUR_PARTITION_SIZE="$(         echo "${CUR_LINE}" | awk '{ print $6 }'  | tr -d "()"  )"
  
        DMCTL_COMMANDS="${DMCTL_COMMANDS}    
linear ${CUR_LOGICAL_START_SECTOR} ${CUR_PARTITION_SIZE} ${SUPER_PARTITION} ${CUR_PARTITION_START_SECTOR}"

        (( CUR_LOGICAL_START_SECTOR = CUR_LOGICAL_START_SECTOR + CUR_PARTITION_SIZE ))

      done <"${TMPFILE1}"

      echo "${DMCTL_COMMANDS}" >"${TMPFILE2}"
      if [ $? -ne 0 ] ; then
        LogError "Error creating the file with the commands for dmctl \"${TMPFILE2}\" - can not create the temporary DM mapper device for the dynamic partition \"${CUR_LOGICAL_PARTITION}\" "
        continue
      fi

      CUR_OUTPUT="$( ${DEBUG_PREFIX} ; ${DMCTL} -f "${TMPFILE2}" 2>&1 )"
      if [ $? -ne 0 ] ; then
        LogMsg "-" "${CUR_OUTPUT}"
        LogError "Error creating the logical device \"${LOGICAL_DEVICE_NAME}\" "
        CONTINUE=${__FALSE}
        THISRC=${__FALSE}

      else
#
# re-read the partition config
#      
        LogInfo "Retrieving the infos about the new logical device \"${LOGICAL_DEVICE_NAME}\" ..."
        CUR_OUTPUT="$( ${DEBUG_PREFIX} ; ${DMCTL}  info "${LOGICAL_DEVICE_NAME}"    2>&1 )"

        LogInfo "The status of the new logical device \"${LOGICAL_DEVICE_NAME}\" is: " &&
          LogMsg "-" "${CUR_OUTPUT}"

#
# wait some seconds for the device to get ready
#
        NEW_LOGICAL_DEVICE="$( ${DMCTL} getpath "${LOGICAL_DEVICE_NAME}" )"

        LogMsg "The new logical device is \"${NEW_LOGICAL_DEVICE}\" "

        if [ ${DMCTL_CREATE_WAIT_TIME} != 0 ] ; then
          LogMsg "Waiting ${DMCTL_CREATE_WAIT_TIME} second(s) now ..."
          sleep ${DMCTL_CREATE_WAIT_TIME}
        fi
        
        if [ -b  "${NEW_LOGICAL_DEVICE}" ] ; then
          LogMsg "The new device \"${NEW_LOGICAL_DEVICE}\" is ready now"
          
          LogInfo "Device successfully created: " && \
            LogMsg "$( ls -l "${NEW_LOGICAL_DEVICE}" 2>&1 )"

          CREATE_FS=${__TRUE}
        else
          LogError "Creating the new device failed"
          THISRC=${__FALSE}
          CONTINUE=§{__FALSE}
        fi

        get_current_status logical_device_config

      fi         

    fi
  fi

  if [ ${CONTINUE} = ${__TRUE} ] ; then
    if [ ${CREATE_FS} = ${__TRUE} ] ; then

      LogMsg "Creating a filesystem on the logical device \"${LOGICAL_DEVICE_NAME}\" using the executable \"${MKFS}\" ..."

      if [ ${DEFAULT_MKFS_WAIT_TIME} != 0 ] ; then
        LogMsg "Waiting now ${DEFAULT_MKFS_WAIT_TIME} seconds ..."
        sleep ${DEFAULT_MKFS_WAIT_TIME}
      fi
      
      CUR_OUTPUT="$( ${DEBUG_PREFIX} ; ${MKFS} ${CUR_LOGICAL_DEVICE} 2>&1 )"
      if [ $? -ne 0 ] ; then
        LogMsg "-" "${CUR_OUTPUT}"
        LogError "Error creating a filesystem on the logical device \"${LOGICAL_DEVICE_NAME}\" using the executable \"${MKFS}\" "
        CONTINUE=${__FALSE}
        THISRC=${__FALSE}
      fi
    fi
  fi
 
  return ${THISRC}
}


function delete_dynamic_partition {
  typeset THISRC=${__TRUE}
  typeset CONTINUE=${__TRUE}
 
  typeset CUR_OUTPUT=""

  typeset CONTINUE=${__TRUE}
  
  LogMsg "Deleting the dynamic partition ..."

  if [ ${CONTINUE} = ${__TRUE} ] ; then
    if [ "${CUR_MOUNT_CONFIG}"x != ""x ] ; then
      LogError "The partition \"${PARTITION_NAME}\" is still mounted:"
      LogMsg "-" "${CUR_MOUNT_CONFIG}"
      THISRC=${__FALSE}
      CONTINUE=${__FALSE}
    fi
  fi

  if [ ${CONTINUE} = ${__TRUE} ] ; then
    if [ "${CUR_LOGICAL_DEVICE}"x = ""x ] ; then
      LogMsg "The logical device for the partition \"${PARTITION_NAME}\" does not exist"
    else
      LogMsg "Deleting the logical device \"${LOGICAL_DEVICE_NAME}\" ..."
      CUR_OUTPUT="$(  ${DEBUG_PREFIX} ; ${DMCTL} delete ${LOGICAL_DEVICE_NAME} 2>&1 )"
      if [ $? -ne 0 ] ; then
        LogMsg "-" "${CUR_OUTPUT}"
        LogError "Error deleting the logical device \"${LOGICAL_DEVICE_NAME}\" "
        CONTINUE=${__FALSE}
        THISRC=${__FALSE}
      fi
    fi
  fi

  if [ ${CONTINUE} = ${__TRUE} ] ; then
    if [ "${CUR_PARTITION_CONFIG}"x = ""x ] ; then
      LogMsg "The partition \"${PARTITION_NAME}\"  does not exist"
    else
#      LogMsg "Deleting the partition \"${PARTITION_NAME}\" ..."
   
       LogWarning "Deleting a dynamic partition is not yet implemented in this script"
       
       cat <<EOT

To delete the dynamic partition boot the phone into the fastboot mode

adb reboot fastboot

and delete the partition using the command

fastboot delete-logical-partition ${PARTITION_NAME}

Note

To list all logical partitions in fastboot mode use this command:

fastboot getvar all 2>&1 | grep "is-logical:" | grep ":yes"

EOT

    fi
  fi

# re-read the current config
#
  get_current_status logical_device_config partition_config
 
  return ${THISRC}
}

function mount_dynamic_partition {
  typeset THISRC=${__TRUE}
  typeset CONTINUE=${__TRUE}
 
  typeset CUR_OUTPUT=""
  
  LogMsg "Mounting the dynamic partition to \"${MOUNT_POINT}\"  ..."

  if [ ${CONTINUE} = ${__TRUE} ] ; then
    if [ "${CUR_MOUNT_CONFIG}"x != ""x ] ; then
      LogMsg "The partition is already mounted:"
      LogMsg "-" "${CUR_MOUNT_CONFIG}"
      THISRC=${__TRUE}
      CONTINUE=${__FALSE}
    fi
  fi

  if [ ${CONTINUE} = ${__TRUE} ] ; then  
    if [ "${CUR_LOGICAL_DEVICE}"x = ""x ] ; then
      LogMsg "The logical device for the partition does not exist"
      THISRC=${__FALSE}
      CONTINUE=${__FALSE}
    fi
  fi

  if [ ${CONTINUE} = ${__TRUE} ] ; then
    if [ ! -d "${MOUNT_POINT}" ] ; then
      LogMsg "Creating the mount point \"${MOUNT_POINT}\" ..."
      CUR_OUTPUT="$( ${DEBUG_PREFIX} ; mkdir -p "${MOUNT_POINT}" 2>&1 )"
      if [ $? -ne 0 ] ; then
        LogMsg "-" "${CUR_OUTPUT}"
        LogError "Error creating the mount point \"${MOUNT_POINT}\""
        THISRC=${__FALSE}
        CONTINUE=${__FALSE}
      fi
    else
      touch "${MOUNT_POINT}/This_directory_is_currently_NOT_mounted"
    fi
  fi

  if [ ${CONTINUE} = ${__TRUE} ] ; then
    CUR_OUTPUT="$( ${DEBUG_PREFIX} ; ${MOUNT} ${MOUNT_OPTIONS} "${CUR_LOGICAL_DEVICE}" "${MOUNT_POINT}" 2>&1 )"
    if [ $? -ne 0 ] ; then
      LogMsg "-" "${CUR_OUTPUT}"
      LogError "Error mounting \"${CUR_LOGICAL_DEVICE}\" at \"${MOUNT_POINT}\" - probably there is no filesystem on that device"
      THISRC=${__FALSE}
      CONTINUE=${__FALSE}
    else
      LogMsg "\"${MOUNT_POINT}\" successfully mounted"
      LogMsg "-"
      CUR_OUTPUT="$( set -x ; df -h "${MOUNT_POINT}" 2>&1 )"
      LogMsg "-" "${CUR_OUTPUT}"
      LogMsg "-"

#
# get the real path to the mount point
#
      REAL_MOUNT_POINT="$( cd "${MOUNT_POINT}" ; pwd -P )"
      
      THISRC=${__TRUE}
    fi
  fi

# re-read the current config
#
  get_current_status mount_device_config
      
  return ${THISRC}
}


function umount_dynamic_partition {
  typeset THISRC=${__TRUE}
  typeset CONTINUE=${__TRUE}
 
  typeset CUR_OUTPUT=""

  typeset UMOUNT_DONE=${__FALSE}

  LogMsg "Umounting the dynamic partition ..."

  LogMsg "Mounting the dynamic partition to \"${MOUNT_POINT}\"  ..."

  if [ ${CONTINUE} = ${__TRUE} ] ; then
    if [ "${CUR_MOUNT_CONFIG}"x = ""x ] ; then
      LogMsg "The partition is currently not mounted"
      THISRC=${__TRUE}
      CONTINUE=${__FALSE}
    fi
  fi

  if [ ${CONTINUE} = ${__TRUE} ] ; then
    CUR_OUTPUT="$( ${DEBUG_PREFIX} ; ${UMOUNT}  "${MOUNT_POINT}" 2>&1 )"
    if [ $? -ne 0 ] ; then
      LogMsg "-" "${CUR_OUTPUT}"
      LogError "Error umounting the mount point \"${MOUNT_POINT}\" ..."
      THISRC=${__FALSE}
    else

      i=0

      while [ $i -lt ${UMOUNT_WAIT_TIME} ] ; do 
        (( i = i + 1 ))

        CUR_MOUNT_DEVICE="$( ${DEBUG_PREFIX} ;  df -h "${MOUNT_POINT}" 2>/dev/null | tail -1 | awk '{ print $1 }' )"
    
        CUR_BASE_DIR_MOUNT_DEVICE="$( ${DEBUG_PREFIX} ;  df -h "${MOUNT_POINT%/*}" 2>/dev/null | tail -1 | awk '{ print $1 }' )"
     
        if [ "${CUR_MOUNT_DEVICE}"x = "${CUR_BASE_DIR_MOUNT_DEVICE}"x ] ; then
          UMOUNT_DONE=${__TRUE}
          break
        fi
        
        [ $i = 0 ] && LogMsg "Waiting up to ${UMOUNT_WAIT_TIME} seconds now ..."

        printf "."
        sleep 1
      done
      printf "\n"
       
      if [ ${UMOUNT_DONE} = ${__TRUE} ] ;then
         LogMsg "\"${MOUNT_POINT}\"  successfully umounted after ${i} seconds"
        THISRC=${__TRUE}
      else
        LogError "Error umounting the mount point \"${MOUNT_POINT}\" "

        CONTINUE=${__FALSE}
        THISRC=${__FALSE}
      fi      
    fi
  fi

  get_current_status mount_device_config
  
  return ${THISRC}
}


function set_permissions_for_the_mount_point {
  typeset THISRC=${__TRUE}

  if [ ! -d "${MOUNT_POINT}" ] ; then
    LogError "The directory \"${MOUNT_POINT}\" does not exist"
    THISRC=${__FALSE}
  else
    LogMsg "Changing the owner and group for the mount point \"${MOUNT_POINT}\" to \"${MOUNT_POINT_OWNER}:${MOUNT_POINT_GROUP}\" ..."
    CUR_OUTPUT="$( ${DEBUG_PREFIX} ; chown ${MOUNT_POINT_OWNER}:${MOUNT_POINT_GROUP} "${MOUNT_POINT}" 2>&1 )"
    if [ $? -gt 1 ] ; then
      LogMsg "-" "${CUR_OUTPUT}"
      LogError "Error changing the owner and group for the mount point \"${MOUNT_POINT}\" to \"${MOUNT_POINT_OWNER}:${MOUNT_POINT_GROUP}\" "
      THISRC=${__FALSE}
      CONTINUE=${__FALSE}
    fi
  
    LogMsg "Setting the permissions for the mount point \"${MOUNT_POINT}\" to \"${MOUNT_POINT_PERMISSIONS}\" ..."
    CUR_OUTPUT="$( ${DEBUG_PREFIX} ; chmod ${MOUNT_POINT_PERMISSIONS} "${MOUNT_POINT}" 2>&1 )"
    if [ $? -gt 1 ] ; then
      LogMsg "-" "${CUR_OUTPUT}"
      LogError "Error setting the permissions for the mount point \"${MOUNT_POINT}\" to \"${MOUNT_POINT_PERMISSIONS}\" "
      THISRC=${__FALSE}
      CONTINUE=${__FALSE}
    fi
  
    LogMsg "Setting the SELinux context for the mount point \"${MOUNT_POINT}\" to \"${MOUNT_POINT_SELINUX_CONTEXT}\" ..."
    
#    CUR_OUTPUT="$( ${DEBUG_PREFIX} ; /data/adb/magisk/busybox chcon -R ${MOUNT_POINT_SELINUX_CONTEXT} "${MOUNT_POINT}" 2>&1 )"
    CUR_OUTPUT="$( ${DEBUG_PREFIX} ; chcon -R ${MOUNT_POINT_SELINUX_CONTEXT} "${MOUNT_POINT}" 2>&1 )"

    if [ $? -gt 1 ] ; then
      LogMsg "-" "${CUR_OUTPUT}"
      LogError "Error setting the SELinux context for the mount point \"${MOUNT_POINT}\" to \"${MOUNT_POINT_SELINUX_CONTEXT}\" "
      THISRC=${__FALSE}
      CONTINUE=${__FALSE}
    fi
  
    LogMsg "-"
    LogMsg "The permissions for the mount point \"${MOUNT_POINT}\" are now:"
    LogMsg "-"
    LogMsg "-" "$( ls -lZd "${MOUNT_POINT}" 2>&1 )"
    LogMsg "-"
  fi
  
  return ${THISRC}
}

function status_dynamic_partition {
  typeset THISRC=${__TRUE}
  typeset CONTINUE=${__TRUE}
 
  typeset CUR_OUTPUT=""
  
  LogMsg "Retrieving the status of the dynamic partition ..."
  LogMsg "-"

  if [ "${DISK_DEVICE}"x != ""x -a "${PARTED}"x != ""x -a ${VERBOSE} != ${__FALSE} ] ; then
    LogMsg "The partition usage on the disk device \"${DISK_DEVICE}\" used for the partition \"${SUPER_PARTITION_NAME}\" is:"
    LogMsg "-"
    ${PARTED} -s "${DISK_DEVICE}"  "unit MB print free"     
    LogMsg "-"
  fi
    
  if [ "${CUR_PARTITION_CONFIG}"x != ""x ] ; then
    LogMsg "The config of the dynamic partition is:"
    LogMsg "-"
    LogMsg "-" "${CUR_PARTITION_CONFIG}"
    LogMsg "-"
  else
    LogMsg "There is no dynamic partition with the name \"${PARTITION_NAME}\" defined"  
  fi
  
  if [ "${CUR_LOGICAL_DEVICE_CONFIG}"x != ""x ] ; then
    LogMsg "The config of the logical Device is:"
    LogMsg "-"
    LogMsg "-" "${CUR_LOGICAL_DEVICE_CONFIG}"
    LogMsg "-"

    if [ ${VERBOSE} = ${__TRUE} ] ; then  
      CUR_OUTPUT="$( ${DMCTL} table "${LOGICAL_DEVICE_NAME}" 2>&1 )"
      LogMsg "The logical extends for the partition are:"
    LogMsg "-"
      LogMsg "-" "${CUR_OUTPUT}"
      LogMsg "-"
    fi
      
  else
    LogMsg "There is no logical device with the name \"${LOGICAL_DEVICE_NAME}\" defined"  
  
  fi

  if [ -d "${MOUNT_POINT}" ] ; then
    if [ "${CUR_MOUNT_CONFIG}"x != ""x ] ; then
      LogMsg "The mount point is mounted:"
      LogMsg "-"
      LogMsg "-" "${CUR_MOUNT_CONFIG}"
      LogMsg "-"
      if [ "${CUR_MOUNT_CONFIG}"x != ""x ] ; then
        LogMsg "The config for the mount point is:"
        LogMsg "-"
        LogMsg "-" "${CUR_MOUNT_POINT_PERMISSIONS}"
      fi

      if [ "${REAL_MOUNT_POINT}" != ""x ] ; then
        LogInfo "-" "-"
        LogInfo "The real mount point is \"${REAL_MOUNT_POINT}\" "
      fi

    else
      LogMsg "The mount point \"${MOUNT_POINT}\" is not mounted"  
    fi
  else
    LogMsg "The mount point \"${MOUNT_POINT}\" does not exist"  
  fi
  LogMsg "-"

  if [ ${VERBOSE} = ${__TRUE} ] ; then  
     
    LogMsg "-"
    LogMsg "The dynamic partitions in the partition \"${SUPER_PARTITION_NAME}\" for the active slot \"${CURRENT_SLOT}\" are: "

      CUR_OUTPUT="$( ${LPDUMP} -s ${CURRENT_SLOT}  | grep ${SUPER_PARTITION_NAME}:  )"
      LogMsg "-"
      LogMsg "-" "${CUR_OUTPUT}"
      LogMsg "-"

      SLOT_DYNAMIC_PARTITION_USAGE_IN_ACTIVE_SLOT="$( echo "( $( ${LPDUMP} -s ${CURRENT_SLOT} | grep super: | grep -v -- "-cow" | grep -v "cow:" | awk '{ print $6 }' | tr -d "(" | tr "\n" "+" ) 0 ) / 2 / 1024  "  | bc )"  
      LogMsg "There are about ${SLOT_DYNAMIC_PARTITION_USAGE_IN_ACTIVE_SLOT} MB used in the partition \"${SUPER_PARTITION_NAME}\" for the partitions of the slot \"${CURRENT_SLOT}\" "
                                
      CUR_OUTPUT="$( ${LPDUMP} -s ${INACTIVE_SLOT}  | grep ${SUPER_PARTITION_NAME}:  )"
      echo "${CUR_OUTPUT}" | grep ${INACTIVE_SLOT} >/dev/null
      if [ $? -eq 0 ] ; then
        LogMsg "-"
        LogMsg "The dynamic partitions in the partition \"${SUPER_PARTITION_NAME}\" for the inactive slot \"${INACTIVE_SLOT}\" are: "

        LogMsg "-"
        LogMsg "-" "${CUR_OUTPUT}"
        LogMsg "-"

        SLOT_DYNAMIC_PARTITION_USAGE_IN_INACTIVE_SLOT="$( echo "( $( ${LPDUMP} -s ${INACTIVE_SLOT} | grep super: | grep -v -- "-cow" | awk '{ print $6 }' | tr -d "(" | tr "\n" "+" ) 0 ) / 2 / 1024  "  | bc )"  
        LogMsg "There are about ${SLOT_DYNAMIC_PARTITION_USAGE_IN_INACTIVE_SLOT} MB used in the partition \"${SUPER_PARTITION_NAME}\" for the partitions of the slot \"${INACTIVE_SLOT}\" "
        LogMsg "-"
      else
        LogMsg "-"
        LogMsg "Can not retrieve the list of dynamic partitions for the inactive slot in the partition \"${SUPER_PARTITION_NAME}\" "
        LogMsg "-"
      fi

      SIZE_OF_SUPER_PARTITION="$( blockdev --getsize64 "${SUPER_PARTITION}" 2>/dev/null )"
      if [ "${SIZE_OF_SUPER_PARTITION}"x != ""x ] ; then
        SIZE_OF_SUPER_PARTITION="$( echo "${SIZE_OF_SUPER_PARTITION} / 1024 /1024" | bc )"
        LogMsg "The size of the partition \"${SUPER_PARTITION_NAME}\" is ${SIZE_OF_SUPER_PARTITION} MB"
      fi

      if [ "${SIZE_OF_SUPER_PARTITION}"x != ""x -a "${SLOT_DYNAMIC_PARTITION_USAGE_IN_INACTIVE_SLOT}"x != ""x -a "${SLOT_DYNAMIC_PARTITION_USAGE_IN_ACTIVE_SLOT}"x != ""x ] ; then
        FREE_SIZE_IN_SUPER_PARTITION="$( echo "${SIZE_OF_SUPER_PARTITION} - ${SLOT_DYNAMIC_PARTITION_USAGE_IN_ACTIVE_SLOT} - ${SLOT_DYNAMIC_PARTITION_USAGE_IN_INACTIVE_SLOT}" | bc )"
        LogMsg "The free space in the partition \"${SUPER_PARTITION_NAME}\" is about ${FREE_SIZE_IN_SUPER_PARTITION} MB"
        if [ ${FREE_SIZE_IN_SUPER_PARTITION} -lt 0 ] ; then
          LogWarning "Something seems to be wrong in the output of lpdump"
        fi
        
        LogMsg "-"
      fi

  fi

  return ${THISRC}
}

# ---------------------------------------------------------------------

function update_config_file_for_service_script {
  typeset THISRC=${__TRUE}
  
  if [ -r "${SERVICE_SCRIPT_CONFIG_FILE}" ] ; then
    BACKUP_FILE="${SERVICE_SCRIPT_CONFIG_FILE}.$$.bkp"
    LogMsg "Creating a backup of the existing file \"${SERVICE_SCRIPT_CONFIG_FILE}\" in the file \"${BACKUP_FILE}\" ..."
    cp "${SERVICE_SCRIPT_CONFIG_FILE}" "${BACKUP_FILE}"
  fi

  LogMsg "Creating the config file for the service.sh script \"${SERVICE_SCRIPT_CONFIG_FILE}\" ..."

  cat >"${SERVICE_SCRIPT_CONFIG_FILE}" <<EOT
#
# config file for service.sh from the Magisk Module for processing the new dynamic partition
#
# File created at $( date )
#

# name of the logical device
#
LOGICAL_DEVICE_NAME="${LOGICAL_DEVICE_NAME}"

# mount point for the new filesystem
#
MOUNT_POINT="${MOUNT_POINT}"

# filesystem on the new device
#
FILESYSTEM_TYPE="${FILESYSTEM_TYPE}"

MOUNT_OPTIONS="${MOUNT_OPTIONS}"

# time to wait in seconds before mounting the new partition
#
MOUNT_WAIT_TIME="${MOUNT_WAIT_TIME}"

# owner, group, permissions, and SELinux Context for the mountpoint
#
MOUNT_POINT_OWNER="${MOUNT_POINT_OWNER}"
MOUNT_POINT_GROUP="${MOUNT_POINT_GROUP}"
MOUNT_POINT_PERMISSIONS="${MOUNT_POINT_PERMISSIONS}"
MOUNT_POINT_SELINUX_CONTEXT="${MOUNT_POINT_SELINUX_CONTEXT}"

EOT
  [ $? -ne 0 ] && THISRC=${__FALSE}
  
  return ${THISRC}
}


# ---------------------------------------------------------------------

ERROR_MESSAGES="$( print_message_prefix ) ${SCRIPT_NAME} started"

# ---------------------------------------------------------------------

if [ "${CUR_USER}"x != "root"x ] ; then
  NO_HEADER=${__TRUE}
  echo "ERROR: This script must be executed by the user root"
  exit 254
fi

# ---------------------------------------------------------------------

NO_HEADER=${__FALSE}

ERRORS_FOUND=${__FALSE}

THISRC=0

ACTIONS=""

CUR_CONFIG_FILE=""

CONFIG_FILE_READ=${__FALSE}

PRINT_USAGE_HELP=${__FALSE}

CUR_LOGFILE="${DEFAULT_LOGFILE}"


while [ $# -ne 0 ] ; do
  CUR_PARAMETER="$1"
  shift
  
  case ${CUR_PARAMETER} in

    -h | --help )
      PRINT_USAGE_HELP=${__TRUE}
      ;;

    -v | --verbose )
      VERBOSE=${__TRUE}
      DEBUG_PREFIX="set -x"
      ;;

    -k | --keep )
      KEEP_TEMPORARY_FILES=${__TRUE}
      ;;

    +k | ++keep )
      KEEP_TEMPORARY_FILES=${__FALSE}
      ;;

    -y | --yes )
      ASSUME_YES=${__TRUE}
      ;;

    +y | ++yes )
      ASSUME_YES=${__FALSE}
      ;;

    +q | ++quiet )
      VERBOSE=${__FALSE}
      ;;

    -e | --ignore_error)
      IGNORE_ERROR=${__TRUE}
      ;;

    +e | ++ignore_error)
      IGNORE_ERROR=${__FALSE}
      ;;

    *=* )
      eval ${CUR_PARAMETER}
      if [ $? -ne 0 ] ; then
        ERROR_MESSAGES="${ERROR_MESSAGES}
ERROR: Error executing \"${CUR_PARAMETER}\" "

        ERRORS_FOUND=${__TRUE}
      fi
      ;;

    -l ) 
      if [ $# -eq 0 ] ; then
        ERROR_MESSAGES="${ERROR_MESSAGES}
ERROR: The parameter \"-l\" is incomplete"

        ERRORS_FOUND=${__TRUE}
      else
        case $1 in 

          default )
            CUR_LOGFILE="${DEFAULT_LOGFILE}"
            ;;

          none )
            CUR_LOGFILE=""
            ;;
        
          * )    
           CUR_LOGFILE="$1"
           ;;
        esac
        shift
      fi
      ;;

    -f ) 
      if [ $# -eq 0 ] ; then
        ERROR_MESSAGES="${ERROR_MESSAGES}
ERROR: The parameter \"-f\" is incomplete"

        ERRORS_FOUND=${__TRUE}
      else
        case $1 in 

          default )
            CUR_CONFIG_FILE="${DEFAULT_CONFIG_FILE}"
            ;;

          none )
            CUR_CONFIG_FILE="none"
            ;;
        
          * )    
           CUR_CONFIG_FILE="$1"
           ;;
        esac
        shift
      fi
      ;;
    
    update )
      ACTIONS="${ACTIONS} update"
      ;;    

    refresh )
      ACTIONS="${ACTIONS} refresh"
      ;;    

    create )
      ACTIONS="${ACTIONS} create"
      ;;

    delete )
      ACTIONS="${ACTIONS} delete"
      ;;

    mount )
      ACTIONS="${ACTIONS} mount"
      ;;
      
    umount )
      ACTIONS="${ACTIONS} umount"
      ;;

    status )
      ACTIONS="${ACTIONS} status"
      ;;
    
    * ) 
        ERROR_MESSAGES="${ERROR_MESSAGES}
ERROR: Unknown parameter found: \"${CUR_PARAMETER}\" "  

      ERRORS_FOUND=${__TRUE}
      ;;

  esac
  
done


if [ ${PRINT_USAGE_HELP} = ${__TRUE} ] ; then
  grep "^#h#" $0 | cut -c4-
  if [ ${VERBOSE} = ${__TRUE} ] ; then
    echo ""
    echo "Default values hardcoded in the script (these values are used if no config file is used):"
    echo ""
    grep "^DEFAULT_" $0 | cut -f2- -d "_"
    echo ""
  fi
  die 1
fi

if [ "${CUR_LOGFILE}"x != ""x ] ; then
  touch "${CUR_LOGFILE}" 2>/dev/null
  if [ $? -ne 0 ] ; then

    ERROR_MESSAGES="${ERROR_MESSAGES}
WARNING: Can not write to the logfile \"${CUR_LOGFILE}\" -- running without a logfile now"

    CUR_LOGFILE=""
  fi
else
  LogMsg "Running without a log file"
fi

if [ "${ERROR_MESSAGES}"x != ""x ] ; then
  LogMsg "-" "${ERROR_MESSAGES}"
fi



if [ "${ACTIONS}"x = ""x ] ; then

  case ${SCRIPT_NAME} in

    create_dynamic_partition )  # known symbolic link
      ACTIONS="create"
      ;;

    create_and_mount_dynamic_partition )  # known symbolic link
      ACTIONS="create mount"
      ;;

    delete_dynamic_partition )  # known symbolic link
      ACTIONS="delete"
      ;;

    umount_and_delete_dynamic_partition )  # known symbolic link
      ACTIONS="umount delete"
      ;;

    mount_dynamic_partition )  # known symbolic link
      ACTIONS="mount"
      ;;
      
    umount_dynamic_partition ) # known symbolic link 
      ACTIONS="umount"
      ;;

    status_dynamic_partition ) # known symbolic link 
      ACTIONS="status"
      ;;

    process_dynamic_partition.sh )
      LogError "The parameter for the action is missing"
      ERRORS_FOUND=${__TRUE}
      ;;

    * ) 
      LogError "\"${SCRIPT_NAME}\" is an unknown script name --  the known symbolic links for this script are:"
      echo
      grep "# known symbolic link"  $0 | grep -v grep |  awk '{ print $1 }'
      echo
      ERRORS_FOUND=${__TRUE}
      ;;
  esac      
fi

if [ "${CUR_CONFIG_FILE}"x = "none"x ] ; then
  CUR_CONFIG_FILE=""
  LogMsg "Running without a config file,  using the hardcoded default values"
elif [ "${CUR_CONFIG_FILE}"x != ""x ] ; then
  if [ ! -f "${CUR_CONFIG_FILE}" ] ; then
    LogError "The config file \"${CUR_CONFIG_FILE}\" does not exist"
    ERRORS_FOUND=${__TRUE}
  else
    LogMsg "Using the config file \"${CUR_CONFIG_FILE}\" "
  fi
else
  if [ -f "${DEFAULT_CONFIG_FILE}" ] ; then
    CUR_CONFIG_FILE="${DEFAULT_CONFIG_FILE}"
  else
    LogMsg "The default config file \"${DEFAULT_CONFIG_FILE}\" does not exist -- running without a config file and using the hardcoded default values"      
  fi
fi

if [ "${CUR_CONFIG_FILE}"x != ""x ] ; then
  LogMsg "Reading the config file \"${CUR_CONFIG_FILE}\" ..."

  CUR_OUTPUT="$( cat "${CUR_CONFIG_FILE}"  2>&1 )"
  if [ $? -ne 0  ] ; then
    LogMsg "-" "${CUR_OUTPUT}"
    LogError "Error reading the config file \"${CUR_CONFIG_FILE}\" "
    ERRORS_FOUND=${__TRUE}
  else
    CUR_CONFIG_FILE_CONTENTS="$( echo "${CUR_OUTPUT}" | grep -E -v "^$|^#" )"
    if [ "${CUR_CONFIG_FILE_CONTENTS}"x = ""x ] ; then
      LogWarning "The config file \"${CUR_CONFIG_FILE}\" is empty"
    else
      LogMsg "Processing the config file \"${CUR_CONFIG_FILE}\" ..."
      CUR_OUTPUT="$( sh -n -x "${CUR_CONFIG_FILE}" 2>&1 )"
      if [ $? -ne 0 ] ; then
        LogMsg "-" "${CUR_OUTPUT}"
        LogError "Syntax errors found in the config file \"${CUR_CONFIG_FILE}\" "
        ERRORS_FOUND=${__TRUE}
      else
        .  "${CUR_CONFIG_FILE}"
        LogMsg " ... config file processed"
      fi
    fi
    CONFIG_FILE_READ=${__TRUE}    
    
  fi
fi


case " ${ACTIONS} " in 

    *\ create\ * )
      REQUIRED_EXECUTABLES="${REQUIRED_EXECUTABLES} LPADD MKFS"
      ;;

    *\ delete*\  )
      ;;

    *\ mount*\  )
      ;;
      
    *\ umount*\  )
      ;;

    *\ status*\  )
      ;;
esac

LogMsg "Checking if the required executables exist ..."

for CUR_VAR in ${REQUIRED_EXECUTABLES} ; do

  eval "${CUR_VAR}=\"\${${CUR_VAR}:=\${DEFAULT_${CUR_VAR}}}\""
  
  eval "CUR_EXECUTABLE=\"\$${CUR_VAR}\""
  
  CUR_PROG_NAME="$( echo ${CUR_VAR} | tr "A-Z" "a-z" )"

  if [ "${CUR_EXECUTABLE}"x = ""x  ] ; then
    LogError "The executable \"${CUR_PROG_NAME}\" is not available via PATH variable"
    ERRORS_FOUND=${__TRUE}
  elif [ ! -x "${CUR_EXECUTABLE}" ] ; then
    LogError "The executable \"${CUR_PROG_NAME}\" does not exist or is not executable"
    ERRORS_FOUND=${__TRUE}
  else
    LogInfo "OK, the executable \"${CUR_EXECUTABLE}\" exists"
  fi
done


FILESYSTEM_TYPE="${MKFS##*.}"

[ "${FILESYSTEM_TYPE}"x != ""x ] && MOUNT_OPTIONS="-t ${FILESYSTEM_TYPE}" || MOUNT_OPTIONS=""

if [ "${GETPROP}"x = ""x ] ; then
  LogError "This script must be executed on an phone running Android"
  ERRORS_FOUND=${__TRUE}
fi

# ---------------------------------------------------------------------

CURRENT_SLOT="$( ${GETPROP} ro.boot.slot_suffix )"
[ "${CURRENT_SLOT}"x  = "_a"x ] && INACTIVE_SLOT="_b" || INACTIVE_SLOT="_a"

#
# use the default values if necessary
#
LIST_OF_VARIABLES_WITH_DEFAULT_VALUES="$( grep ^DEFAULT_ $0 | cut -f1 -d "=" )"

for CUR_VAR in ${LIST_OF_VARIABLES_WITH_DEFAULT_VALUES} ; do
  P2="${CUR_VAR#DEFAULT_*}"

  eval $P2="\${$P2:=\${DEFAULT_$P2}}"
done


if [ ! -d "${TMP_DIR}" ] ; then
  LogError "The directory for temporary files, \"${TMP_DIR}\", does not exist"
  ERRORS_FOUND=${__TRUE}
fi  
  
if [[ ${MOUNT_POINT} != /*/* ]] ; then
  LogError "The mount point \"${MOUNT_POINT}\" is not allowed"
  ERRORS_FOUND=${__TRUE}
fi  

# ---------------------------------------------------------------------

if [ ${ERRORS_FOUND} != ${__FALSE} ] ; then
  die 5 "One or more errors found"
fi

# ---------------------------------------------------------------------

if [ -d "${MOUNT_POINT}" ] ; then
  REAL_MOUNT_POINT="$( cd   "${MOUNT_POINT}" ; pwd -P )"
else  
  REAL_MOUNT_POINT="${MOUNT_POINT}"
fi

LogMsg "-"
LogMsg "Using the partition \"${PARTITION_NAME}\" "

if [[ ${PARTITION_SIZE} == /* ]] ; then
  LogMsg "The requested partition size is the size of the file/device \"${PARTITION_SIZE}\" "
else
  LogMsg "The requested partition size is \"${PARTITION_SIZE}\" "
fi
LogMsg "Using the logical device \"${LOGICAL_DEVICE_NAME}\" "
LogMsg "The mount point is \"${MOUNT_POINT}\" "
LogMsg "-"

# get the current tsatus

LogMsg "Retrieving the current status ..."

get_current_status

# ---------------------------------------------------------------------


for CUR_ACTION in ${ACTIONS} ; do

  case ${CUR_ACTION} in 
  
      create )
        if [ ${ASSUME_YES} != ${__TRUE} ] ; then
          echo "${DISCLAIMER}"
          echo 
          echo "(Note: Use the script parameter \"-y\" to suppress this request)"
          echo
          printf "Enter YES<return> to continue : "
          read USER_INPUT
          if [ "${USER_INPUT}"x != "YES"x ] ; then
            die 253 "Script aborted by the user"
          fi
        fi
      
        create_dynamic_partition
        THISRC=$?
        [ ${THISRC} = ${__TRUE} ]  && update_config_file_for_service_script
        ;;
            
      update )
        update_config_file_for_service_script
        THISRC=$?
        ;;
  
      delete )
        delete_dynamic_partition
        THISRC=$?
        [ ${THISRC} = 0 ] && \rm -f "${SERVICE_SCRIPT_CONFIG_FILE}" 2>/dev/null
        ;;
  
      mount )
        if [ "${LAST_ACTION}"x = "create"x -a ${MOUNT_WAIT_TIME} != 0 ] ;then
          LogMsg "Waiting for ${MOUNT_WAIT_TIME} seconds now ..."
          sleep ${MOUNT_WAIT_TIME}
        fi
        
        mount_dynamic_partition && \
          set_permissions_for_the_mount_point || \
          THISRC=${__FALSE}
        ;;
        
      umount )
        umount_dynamic_partition
        THISRC=$?
        ;;
  
      refresh )
        set_permissions_for_the_mount_point
        THISRC=$?
        ;;

      status )
        status_dynamic_partition
        THISRC=$?
        ;;
  
      * )  
       die 254 "INTERNAL ERROR: Unknown action \"${ACTION}\" "
       ;;
  esac

  LAST_ACTION="${CUR_ACTION}"
  
  [ ${THISRC} != ${__TRUE} -a ${IGNORE_ERROR} = ${__FALSE}  ] && break
done


# ---------------------------------------------------------------------

die ${THISRC}

# ---------------------------------------------------------------------
