#
# Customization script for the Magisk Module with git 
#
# History
#   30.10.2024 /bs
#     initial release
#
#
# Notes:
#
# This Magisk Module contains files for arm64 CPUs
#
# Documentation for creating Magisk Modules: https://topjohnwu.github.io/Magisk/guides.html
#
# Environment variables that can be used:
#
#    MAGISK_VER (string): the version string of current installed Magisk (e.g. v20.0)
#    MAGISK_VER_CODE (int): the version code of current installed Magisk (e.g. 20000)
#    BOOTMODE (bool): true if the module is being installed in the Magisk app
#    MODPATH (path): the path where your module files should be installed
#    TMPDIR (path): a place where you can temporarily store files
#    ZIPFILE (path): your module’s installation zip
#    ARCH (string): the CPU architecture of the device. Value is either arm, arm64, x86, or x64
#    IS64BIT (bool): true if $ARCH is either arm64 or x64
#    API (int): the API level (Android version) of the device (e.g. 21 for Android 5.0)
#

# -----------------------------------------------------------------------------

# define constants
#
__TRUE=0
__FALSE=1

# -----------------------------------------------------------------------------
# init global variables
#

MODULE_VERSION="$( grep "^version=" $MODPATH/module.prop  | cut -f2 -d "=" )"

MODULE_NAME="$( grep "^id=" $MODPATH/module.prop  | cut -f2 -d "=" )"

MODULE_DESC="${MODULE_NAME} ${MODULE_VERSION}"

GIT_CONFIG_DIR="/data/local/tmp/sysroot/etc"

GIT_CONFIG_FILE="${GIT_CONFIG_DIR}/gitconfig"
GIT_ATTRIBUTES_FILE="${GIT_CONFIG_DIR}/gitattributes"


# -----------------------------------------------------------------------------
#
if [ 0 = 1 -o -r /data/local/tmp/debug ] ; then
  LOGFILE="/data/local/tmp/${MODULE_NAME}_customize.log"

  ui_print "Writing all messages to the log file ${LOGFILE}"
  exec 1>${LOGFILE} 2>&1
  set -x
fi

# -----------------------------------------------------------------------------

function LogMsg {
  ui_print "$*"
}

function LogInfo {
  LogMsg "INFO: $*"
}

function LogWarning {
  LogMsg "WARNING: $*"
}

function LogError {
  LogMsg "ERROR: $*"
}


function myprintf {
  if  [  "${INSTALL_PROGRAM}"x = "magisk binary"x ] ; then
    printf "$*"
  fi
}


# -----------------------------------------------------------------------------
#
ps | grep "^$$ "  | grep /debug_ramdisk/.magisk/busybox/busybox >/dev/null
if [ $? -eq 0 ] ; then
  LogMsg "The Module installation is done via the magisk binary"
  INSTALL_PROGRAM="magisk binary"
else
  LogMsg "The Module installation is done via the Magisk GUI"
  INSTALL_PROGRAM="Magisk GUI"
fi


# -----------------------------------------------------------------------------

LogMsg "The current environment for this installation is"

LogMsg "The version of the installed Magisk is \"${MAGISK_VER}\" (${MAGISK_VER_CODE})"

LogInfo "BOOTMODE is \"${BOOTMODE}\" "
LogInfo "MODPATH is \"${MODPATH}\" "
LogInfo "TMPDIR is \"${TMPDIR}\" "
LogInfo "ZIPFILE is \"${ZIPFILE}\" "
LogInfo "ARCH is \"${ARCH}\" "
LogInfo "IS64BIT is \"${IS64BIT}\" "
LogInfo "API is \"${API}\" "

# -----------------------------------------------------------------------------

# example output for the variables:


#  The version of the installed Magisk is "25.0" (25000)
#  INFO: BOOTMODE is "true" 
#  INFO: MODPATH is "/data/adb/modules_update/PlayStore_for_MicroG" 
#  INFO: TMPDIR is "/dev/tmp" 
#  INFO: ZIPFILE is "/data/user/0/com.topjohnwu.magisk/cache/flash/install.zip" 
#  INFO: ARCH is "arm64" 
#  INFO: IS64BIT is "true" 
#  INFO: API is "32"


# -----------------------------------------------------------------------------

LogMsg "Installing the Magisk Module with ${MODULE_NAME} for Android \"${MODULE_VERSION}\" ..."

LogMsg "Checking the OS configuration ..."

ERRORS_FOUND=${__FALSE}

MACHINE_TYPE="$( uname -m )"

# check the current CPU
#
LogMsg "Checking the type of the CPU used in this device ...."

LogMsg "The CPU in this device is a ${ARCH} CPU"
LogMsg "The machine type reported by \"uname -m\" is \"${MACHINE_TYPE}\" "


if [ "${ARCH}"x != "arm64"x ] ; then
  abort "This Magisk module is for arm64 CPUs only"
fi

# ---------------------------------------------------------------------

LogMsg "Installing git for Android ..."

if [ ! -d "${GIT_CONFIG_DIR}" ] ; then
  LogMsg "Creating the directory for the git config files \"${GIT_CONFIG_DIR}\" ..."
  mkdir -p "${GIT_CONFIG_DIR}"
else
  LogMsg "The directory for the git config files  \"${GIT_CONFIG_DIR}\" already exists"
fi

LogMsg "git from this Magisk Module uses these config files:

${GIT_CONFIG_FILE}
${GIT_ATTRIBUTES_FILE}

"

# ----------------------------------------------------------------------

LogMsg "Processing the git config file \"${GIT_CONFIG_FILE}\" ..."

if [ -r "${GIT_CONFIG_FILE}" ] ; then
  LogMsg "The git config file \"${GIT_CONFIG_FILE}\" already exists"
  LogMsg ""
  LogMsg "$( ls -l ${GIT_CONFIG_FILE} 2>&1 )"
  LogMsg ""
else
  LogMsg "Creating an initial git config file \"${GIT_CONFIG_FILE}\" ...."
  cp "${MODPATH}/gitconfig" "${GIT_CONFIG_FILE}"
  chown shell:shll  "${GIT_CONFIG_FILE}"
  LogMsg ""
  LogMsg "$( ls -l ${GIT_CONFIG_FILE} 2>&1 )"
  LogMsg ""

fi

# ---------------------------------------------------------------------
#
LogMsg "Processing the certificate files ..."

SYSROOT_DIR="/system"



CERTIFICATE_BUNDLE_FILE="${SYSROOT_DIR}/usr/share/security/ca-certificates.crt"

NEW_CERTIFICATE_BUNDLE_FILE="${MODPATH}/${CERTIFICATE_BUNDLE_FILE}"

if [ -r "${CERTIFICATE_BUNDLE_FILE}" ] ; then
  LogMsg "The certificate bundle file \"${CERTIFICATE_BUNDLE_FILE}\" already exists:"
  LogMsg ""
  LogMsg "$( ls -l "${CERTIFICATE_BUNDLE_FILE}" 2>&1 )"
  LogMsg ""
  LogMsg "Reusing the existing certificate bundle file \"${CERTIFICATE_BUNDLE_FILE}\" ..."
  cp "${CERTIFICATE_BUNDLE_FILE}" "${NEW_CERTIFICATE_BUNDLE_FILE}"
else
  LogMsg "Creating the certificate bundle file \"${NEW_CERTIFICATE_BUNDLE_FILE}\" ..."

  ls -l ${SYSROOT_DIR}/etc/security/cacerts*/*.0 2>/dev/null 1>/dev/null
  if  [ $? -ne 0 ] ; then
    LogWarning "No certificates found in the directories \"${SYSROOT_DIR}/etc/security/cacerts*\" -- can not create the bundle file \"${NEW_CERTIFICATE_BUNDLE_FILE}\""
  else
    for i in  ${SYSROOT_DIR}/etc/security/cacerts*/*.0; do
      myprintf "."
      echo "$(sed -n "/BEGIN CERTIFICATE/,/END CERTIFICATE/p" $i)" >>"${NEW_CERTIFICATE_BUNDLE_FILE}"
    done
    LogMsg ""    
    LogMsg "... done:"
    LogMsg ""
    LogMsg "$( ls -l "${NEW_CERTIFICATE_BUNDLE_FILE}" 2>&1 )"
    LogMsg ""

  fi
fi

# ----------------------------------------------------------------------
#
LogMsg "Correcting the permissions for the new files ..."

cd "${MODPATH}"

# first the directories with data and config files 
# 
for i in etc include usr ; do

  [[ $i != /\* ]] && CUR_ENTRY="${MODPATH}/system/$i" || CUR_ENTRY="$i"

  [ ! -d "${CUR_ENTRY}" ] && continue

  LogMsg "Processing the files in the directory ${CUR_ENTRY} ..."
  set_perm_recursive "${CUR_ENTRY}" 0 0 0755 0644 u:object_r:system_file:s0
  chcon -R -h u:object_r:system_file:s0 "${CUR_ENTRY}"
done


# now the directories with binaries
# 
for i in bin lib lib64 usr/bin usr/libexec ; do

  [[ $i != /\* ]] && CUR_ENTRY="${MODPATH}/system/$i" || CUR_ENTRY="$i"
	
  [ ! -d "${CUR_ENTRY}" ] && continue

  LogMsg "Processing the files in the directory ${CUR_ENTRY} ..."

  set_perm_recursive "${CUR_ENTRY}" 0 0 0755 0755 u:object_r:system_file:s0

  chcon -R -h  u:object_r:system_file:s0 "${CUR_ENTRY}"
done


