#
# Customization script for the Magisk Module with SELinux tools 
#
# History
#   23.07.2024 /bs
#     initial release
#   20.08.2024 /bs
#     minor changes
#
# Notes:
#
# This Magisk Module contains SELinux tools for ARM, MIPS, and X86 CPUs.
#
# The binaries are downloaded from this repository: https://github.com/xmikos/setools-android
#
# Documentation for creating Magisk Modules: https://topjohnwu.github.io/Magisk/guides.html
#
# Environment variables that can be used:
#
#    MAGISK_VER (string): the version string of current installed Magisk (e.g. v20.0)
#    MAGISK_VER_CODE (int): the version code of current installed Magisk (e.g. 20000)
#    BOOTMODE (bool): true if the module is being installed in the Magisk app
#    MODPATH (path): the path where your module files should be installed
#    TMPDIR (path): a place where you can temporarily store files
#    ZIPFILE (path): your module’s installation zip
#    ARCH (string): the CPU architecture of the device. Value is either arm, arm64, x86, or x64
#    IS64BIT (bool): true if $ARCH is either arm64 or x64
#    API (int): the API level (Android version) of the device (e.g. 21 for Android 5.0)
#

# -----------------------------------------------------------------------------

# define constants
#
__TRUE=0
__FALSE=1


# -----------------------------------------------------------------------------
# init global variables
#
MODULE_VERSION="$( grep "^version=" $MODPATH/module.prop  | cut -f2 -d "=" )"


FORCE_SEMAPHOR_FILE="/sdcard/Download/force_installation_of_selinux_tools"


# -----------------------------------------------------------------------------

function LogMsg {
  ui_print "$*"
}

function LogInfo {
  LogMsg "INFO: $*"
}

function LogWarning {
  LogMsg "WARNING: $*"
}

function LogError {
  LogMsg "ERROR: $*"
}

# returns: ${__TRUE}  - semaphor exists
#          ${__FALSE} - semaphor does NOT exist
#
function check_for_force_semaphor {
  LOCAL_THISRC=${__FALSE}

  if [ "${FORCE_SEMAPHOR_FILE}"x != ""x  ] ; then
    if [ -r "${FORCE_SEMAPHOR_FILE}" ] ; then
       LogInfo "The file \"${FORCE_SEMAPHOR_FILE}\" exists -- the installation will continue"
       LOCAL_THISRC=${__TRUE}
    else
      LogMsg "Create the file \"${FORCE_SEMAPHOR_FILE}\" before installing this Magisk Module to force the installation"
    fi
  fi

 return ${LOCAL_THISRC}
}

# -----------------------------------------------------------------------------

LogMsg "The current environment for this installation is"

LogMsg "The version of the installed Magisk is \"${MAGISK_VER}\" (${MAGISK_VER_CODE})"

LogInfo "BOOTMODE is \"${BOOTMODE}\" "
LogInfo "MODPATH is \"${MODPATH}\" "
LogInfo "TMPDIR is \"${TMPDIR}\" "
LogInfo "ZIPFILE is \"${ZIPFILE}\" "
LogInfo "ARCH is \"${ARCH}\" "
LogInfo "IS64BIT is \"${IS64BIT}\" "
LogInfo "API is \"${API}\" "

# -----------------------------------------------------------------------------

# example output for the variables:


#  The version of the installed Magisk is "25.0" (25000)
#  INFO: BOOTMODE is "true" 
#  INFO: MODPATH is "/data/adb/modules_update/PlayStore_for_MicroG" 
#  INFO: TMPDIR is "/dev/tmp" 
#  INFO: ZIPFILE is "/data/user/0/com.topjohnwu.magisk/cache/flash/install.zip" 
#  INFO: ARCH is "arm64" 
#  INFO: IS64BIT is "true" 
#  INFO: API is "32"


# -----------------------------------------------------------------------------
LogMsg "-"

LogMsg "Installing the Magisk Module with the SELinux tools \"${MODULE_VERSION}\" ..."

# LogMsg "Checking the OS configuration ..."

ERRORS_FOUND=${__FALSE}

MACHINE_TYPE="$( uname -m )"

# check the current CPU
#
LogMsg "Checking the type of the CPU used in this device ...."
LogMsg "The CPU in this device is a ${ARCH} CPU"
LogMsg "The machine type reported by \"uname -m\" is \"${MACHINE_TYPE}\" "

if [ ${ERRORS_FOUND} = ${__TRUE} ] ; then
  check_for_force_semaphor || abort "ERRROR: One or more errors in the prerequisite checks found"
fi

SOURCE_DIR=""

case ${ARCH} in

  arm64 )
    SOURCE_DIR="arm64-v8a"
    ;;

  arm )
    SOURCE_DIR="armeabi-v7a"
    ;;

  mips64 )
    SOURCE_DIR="mips"
    ;;

  x86 )
    SOURCE_DIR="x86"
    ;;

  x86_64 )
    SOURCE_DIR="x86_64"
    ;;

  * )
    case ${MACHINE_TYPE} in

      aarch64 )
        SOURCE_DIR="arm64-v8a"
        ;;

      armv7l )
        SOURCE_DIR="armeabi-v7a"
        ;;

      i386 | i686 )
        SOURCE_DIR="x86"
        ;;
    
      * )
        LogWarning "The CPU seems to be unknown ... now searching for the executables usable by this CPU ..."
    
        cd "${MODPATH}/setools-android"

        for CUR_DIR in * ; do
          LogMsg "Testing the binaries in the directory \"${CUR_DIR}\" ..."
          CUR_OUTPUT="$( LD_LIBRARY_PATH=${CUR_DIR}/lib64:${CUR_DIR}/lib  ${CUR_DIR}/bin/seinfo 2>&1 )"
          if [[ "${CUR_OUTPUT}" != *not\ executable* ]] ; then
            LogMsg "The binaries from the directory \"${CUR_DIR}\" seems to be usable"
            SOURCE_DIR="${CUR_DIR}"
            break
          fi
        done
        ;;
    esac
esac

if [ "${SOURCE_DIR}"x = ""x ] ; then
  abort "No executables for the current CPU found"
else
  LogMsg "Using the binaries from the directory \"${SOURCE_DIR}\" "

  cp -r  "${MODPATH}/setools-android/${SOURCE_DIR}/"* "${MODPATH}/system"

  if [ -d "${MODPATH}/system/bin"  ] ; then
    set_perm_recursive "${MODPATH}/system/bin" 0 0 0755 0555 u:object_r:system_file:s0
    chcon -R -h  u:object_r:system_file:s0 "${MODPATH}/system/bin"
  fi
  if [ -d  "${MODPATH}/system/lib64" ] ; then
    set_perm_recursive "${MODPATH}/system/lib64" 0 0 0755 0644  u:object_r:system_lib_file:s0
    chcon -R -h  u:object_r:system_lib_file:s0  "${MODPATH}/system/lib64"
  fi

  if [ -d  "${MODPATH}/system/lib" ] ; then
    set_perm_recursive "${MODPATH}/system/lib" 0 0 0755 0644  u:object_r:system_lib_file:s0
    chcon -R -h  u:object_r:system_lib_file:s0  "${MODPATH}/system/lib"
  fi

fi
