#
# Customization script for the Magisk Module with the tools to process dynamic partitions
#
# History
#   01.08.2024 /bs
#     initial release
#     
#
# Notes:
#
# see here for details about these executables:
#
# https://xdaforums.com/t/magisk-twrp-arm32-64-a8-universal-read-only-to-read-write-for-android-ro2rw-auto-converting-super-system-partitions-to-read-write-mode.4521131/ 
#
# This Magisk Module contains binaries for arm and arm64 CPUs
#
# Documentation for creating Magisk Modules: https://topjohnwu.github.io/Magisk/guides.html
#
# Environment variables that can be used:
#
#    MAGISK_VER (string): the version string of current installed Magisk (e.g. v20.0)
#    MAGISK_VER_CODE (int): the version code of current installed Magisk (e.g. 20000)
#    BOOTMODE (bool): true if the module is being installed in the Magisk app
#    MODPATH (path): the path where your module files should be installed
#    TMPDIR (path): a place where you can temporarily store files
#    ZIPFILE (path): your module’s installation zip
#    ARCH (string): the CPU architecture of the device. Value is either arm, arm64, x86, or x64
#    IS64BIT (bool): true if $ARCH is either arm64 or x64
#    API (int): the API level (Android version) of the device (e.g. 21 for Android 5.0)
#

# -----------------------------------------------------------------------------

# define constants
#
__TRUE=0
__FALSE=1


# -----------------------------------------------------------------------------
# init global variables
#

#MODULE_VERSION="9.1.0.0"

MODULE_VERSION="$( grep "^version=" $MODPATH/module.prop  | cut -f2 -d "=" )"

FORCE_SEMAPHOR_FILE="/sdcard/Download/force_installation_of_ro2rwtools"

BINARY=""

# -----------------------------------------------------------------------------

function LogMsg {
  ui_print "$*"
}

function LogInfo {
  LogMsg "INFO: $*"
}

function LogWarning {
  LogMsg "WARNING: $*"
}

function LogError {
  LogMsg "ERROR: $*"
}

# returns: ${__TRUE}  - semaphor exists
#          ${__FALSE} - semaphor does NOT exist
#
function check_for_force_semaphor {
  LOCAL_THISRC=${__FALSE}

  if [ "${FORCE_SEMAPHOR_FILE}"x != ""x  ] ; then
    if [ -r "${FORCE_SEMAPHOR_FILE}" ] ; then
       LogInfo "The file \"${FORCE_SEMAPHOR_FILE}\" exists -- the installation will continue"
       LOCAL_THISRC=${__TRUE}
    else
      LogMsg "Create the file \"${FORCE_SEMAPHOR_FILE}\" before installing this Magisk Module to force the installation"
    fi
  fi

 return ${LOCAL_THISRC}
}

# -----------------------------------------------------------------------------

LogMsg "The current environment for this installation is"

LogMsg "The version of the installed Magisk is \"${MAGISK_VER}\" (${MAGISK_VER_CODE})"

LogInfo "BOOTMODE is \"${BOOTMODE}\" "
LogInfo "MODPATH is \"${MODPATH}\" "
LogInfo "TMPDIR is \"${TMPDIR}\" "
LogInfo "ZIPFILE is \"${ZIPFILE}\" "
LogInfo "ARCH is \"${ARCH}\" "
LogInfo "IS64BIT is \"${IS64BIT}\" "
LogInfo "API is \"${API}\" "

# -----------------------------------------------------------------------------

# example output for the variables:


#  The version of the installed Magisk is "25.0" (25000)
#  INFO: BOOTMODE is "true" 
#  INFO: MODPATH is "/data/adb/modules_update/PlayStore_for_MicroG" 
#  INFO: TMPDIR is "/dev/tmp" 
#  INFO: ZIPFILE is "/data/user/0/com.topjohnwu.magisk/cache/flash/install.zip" 
#  INFO: ARCH is "arm64" 
#  INFO: IS64BIT is "true" 
#  INFO: API is "32"


# -----------------------------------------------------------------------------

LogMsg "Installing the Magisk Module with tools to process dynamic partitions version \"${MODULE_VERSION}\" ..."

# LogMsg "Checking the OS configuration ..."

ERRORS_FOUND=${__FALSE}

MACHINE_TYPE="$( uname -m )"

DATA_BASE_DIR="/data/local/tmp"

CUR_DATA_DIR=""

#CUR_DATA_DIR="${DATA_BASE_DIR}/share/vim"
#CUR_DATA_TAR_ARCHIVE="${MODPATH}/share.tar.gz"

# check the current CPU
#
LogMsg "Checking the type of the CPU used in this device ...."
LogMsg "The CPU in this device is a ${ARCH} CPU"
LogMsg "The machine type reported by \"uname -m\" is \"${MACHINE_TYPE}\" "

if [ ${ERRORS_FOUND} = ${__TRUE} ] ; then
  check_for_force_semaphor || abort "ERROR: One or more errors in the prerequisite checks found"
fi

if [ "${ARCH}"x = "arm64"x ] ; then
  SOURCE_DIR="${MODPATH}/ro2rw_executables/arm64"
elif [ "${ARCH}"x = "arm"x ] ; then
  SOURCE_DIR="${MODPATH}/ro2rw_executables/arm32"
else
  abort "This Magisk module is for arm CPUs only"
fi

TARGET_DIR="${MODPATH}/system/bin/ro2rw"

LogMsg "Copying the files from \"${SOURCE_DIR}\" to \"${TARGET_DIR}\" ..."

mkdir -p "${TARGET_DIR}" && cp -r "${SOURCE_DIR}/"* "${TARGET_DIR}" 

LogMsg "Correcting the permissions for the new binaries ..."

set_perm_recursive "${MODPATH}/system/bin" 0 0 0755 0755 u:object_r:system_file:s0
set_perm_recursive "${MODPATH}/system/lib64" 0 0 0755 0755 u:object_r:system_lib_file:s0

# change the SELinux context for symbolic links also
#
chcon -R -h  u:object_r:system_file:s0 "${MODPATH}/system/bin"
chcon -R -h  u:object_r:system_lib_file:s0 "${MODPATH}/system/lib64"

if [ ! -z  ${CUR_DATA_DIR} ] ; then
  if [ -d "${CUR_DATA_DIR}" ] ; then 
    LogMsg "The data directory \"${CUR_DATA_DIR}\" already exists"
  else
    LogMsg "Creating the data directory \"${CUR_DATA_DIR}\" ..."
    if [ ! -r "${CUR_DATA_TAR_ARCHIVE}" ] ; then
      LogError "The tar file with the contents for the data dir \"${CUR_DATA_TAR_ARCHIVE}\" does not exist"
    elif [ ! -d "${DATA_BASE_DIR}" ]; then
      LogError "The directory for the data files \"${DATA_BASE_DIR}\" does not exist"
    else    
      cd "${DATA_BASE_DIR}" && tar -xzf "${CUR_DATA_TAR_ARCHIVE}" || \
        LogWarning "Error extracting the file \"${CUR_DATA_TAR_ARCHIVE}\" "
      set_perm_recursive "${DATA_BASE_DIR}/share" 2000 2000 0755 0644 u:object_r:system_file:s0
    fi
  fi
fi

