#
# Customization script for the Magisk Module perl 5.43 
#
# History
#   15.05.2025 /bs
#     initial release
#   07.07.2025 /bs
#     the script now sets the variable INSTALL_MODE to "update" if the module is already installed
#
#
# Notes:
#
# This Magisk Module contains files for arm64 CPUs
#
# Documentation for creating Magisk Modules: https://topjohnwu.github.io/Magisk/guides.html
#
# Environment variables that can be used:
#
#    MAGISK_VER (string): the version string of current installed Magisk (e.g. v20.0)
#    MAGISK_VER_CODE (int): the version code of current installed Magisk (e.g. 20000)
#    BOOTMODE (bool): true if the module is being installed in the Magisk app
#    MODPATH (path): the path where your module files should be installed
#    TMPDIR (path): a place where you can temporarily store files
#    ZIPFILE (path): your module’s installation zip
#    ARCH (string): the CPU architecture of the device. Value is either arm, arm64, x86, or x64
#    IS64BIT (bool): true if $ARCH is either arm64 or x64
#    API (int): the API level (Android version) of the device (e.g. 21 for Android 5.0)
#

# -----------------------------------------------------------------------------

DISCLAIMER="
to be written
"

# -----------------------------------------------------------------------------

# define constants
#
__TRUE=0
__FALSE=1

# -----------------------------------------------------------------------------
# init global variables
#

MODULE_VERSION="$( grep "^version=" $MODPATH/module.prop  | cut -f2 -d "=" )"

MODULE_NAME="$( grep "^id=" $MODPATH/module.prop  | cut -f2 -d "=" )"

MODULE_DESC="${MODULE_NAME} ${MODULE_VERSION}"


# -----------------------------------------------------------------------------
#
if [ 0 = 1 -o -r /data/local/tmp/debug ] ; then
  LOGFILE="/data/local/tmp/${MODULE_NAME}_customize.log"

  ui_print "Writing all messages to the log file ${LOGFILE}"
  exec 1>${LOGFILE} 2>&1

  if [ 0 = 1 -o -r /data/local/tmp/trace ] ; then
    set -x
  fi
fi


# -----------------------------------------------------------------------------
#
if [ -d "/data/adb/modules/${MODULE_NAME}/system" ] ; then
  LogMsg "This is an update of an already installed module"
  INSTALL_MODE="update"
else
  LogMsg "This is a new installation of this module"
  INSTALL_MODE="install"
fi


# -----------------------------------------------------------------------------

#For convenience, you can also declare a list of folders you want to replace in the variable name REPLACE. The module installer script will create the .replace file into the folders listed in REPLACE. For example:
# 
# REPLACE="
# /system/app/YouTube
# /system/app/Bloatware
# "

# The list above will result in the following files being created: $MODPATH/system/app/YouTube/.replace and $MODPATH/system/app/Bloatware/.replace.
# 
# For convenience, you can also declare a list of files/folders you want to remove in the variable name REMOVE. The module installer script will create the corresponding dummy devices. For example:
# 
# REMOVE="
# /system/app/YouTube
# /system/fonts/Roboto.ttf
# "
# 

# -----------------------------------------------------------------------------

function LogMsg {
  ui_print "$*"
}

function LogInfo {
  LogMsg "INFO: $*"
}

function LogWarning {
  LogMsg "WARNING: $*"
}

function LogError {
  LogMsg "ERROR: $*"
}


# -----------------------------------------------------------------------------

LogMsg "The current environment for this installation is"

LogMsg "The version of the installed Magisk is \"${MAGISK_VER}\" (${MAGISK_VER_CODE})"

LogInfo "BOOTMODE is \"${BOOTMODE}\" "
LogInfo "MODPATH is \"${MODPATH}\" "
LogInfo "TMPDIR is \"${TMPDIR}\" "
LogInfo "ZIPFILE is \"${ZIPFILE}\" "
LogInfo "ARCH is \"${ARCH}\" "
LogInfo "IS64BIT is \"${IS64BIT}\" "
LogInfo "API is \"${API}\" "

# -----------------------------------------------------------------------------

# example output for the variables:


#  The version of the installed Magisk is "25.0" (25000)
#  INFO: BOOTMODE is "true" 
#  INFO: MODPATH is "/data/adb/modules_update/PlayStore_for_MicroG" 
#  INFO: TMPDIR is "/dev/tmp" 
#  INFO: ZIPFILE is "/data/user/0/com.topjohnwu.magisk/cache/flash/install.zip" 
#  INFO: ARCH is "arm64" 
#  INFO: IS64BIT is "true" 
#  INFO: API is "32"

# -----------------------------------------------------------------------------

LogMsg "Installing the Magisk Module with ${MODULE_NAME} for Android \"${MODULE_VERSION}\" ..."

# -----------------------------------------------------------------------------
#

LogMsg "Retrieving the perl home directory to use ..."

PERL_HOME_DIR="$( grep "^PERL_HOME_DIR=" $MODPATH/system/bin/perl_env  | tail -1 | cut -f2 -d '"' )"
if [ "${PERL_HOME_DIR}"x = ""x ] ; then
  LogWarning "No value for PERL_HOME_DIR found in the file \"$MODPATH/system/bin/perl_env\" - now using the default value"
  PERL_HOME_DIR="/data/local/tmp/perl_home"
fi

LogMsg "Using the perl home directory \"${PERL_HOME_DIR}\" "


CPAN_CONFIG_FILE_FILENAME="MyConfig.pm"

CPAN_CONFIG_FILE="${PERL_HOME_DIR}/.cpan/CPAN/${CPAN_CONFIG_FILE_FILENAME}"

CPAN_CONFIG_FILE_SOURCE="${MODPATH}/${CPAN_CONFIG_FILE_FILENAME}"

INSTALL_CPAN_CONFIG_FILE=${__TRUE}

# -----------------------------------------------------------------------------

LogMsg "Installing the Magisk Module with ${MODULE_NAME} for Android \"${MODULE_VERSION}\" ..."

LogMsg "Checking the OS configuration ..."

ERRORS_FOUND=${__FALSE}

MACHINE_TYPE="$( uname -m )"

# check the current CPU
#
LogMsg "Checking the type of the CPU used in this device ...."

LogMsg "The CPU in this device is a ${ARCH} CPU"
LogMsg "The machine type reported by \"uname -m\" is \"${MACHINE_TYPE}\" "


if [ "${ARCH}"x != "arm64"x ] ; then
  abort "This Magisk module is for arm64 CPUs only"
fi

# ---------------------------------------------------------------------

LogMsg "Installing the Magisk Module perl 5.43 ..."

# ---------------------------------------------------------------------
#
if [ -d ${PERL_HOME_DIR} ] ; then
  LogMsg "The Perl home directory \"${PERL_HOME_DIR}\" already exists"
else
  LogMsg "Creating the Perl home directory \"${PERL_HOME_DIR}\" ..."
  mkdir -p "${PERL_HOME_DIR}" 
  mkdir -p "${PERL_HOME_DIR}/lib/perl5"
  mkdir -p "${PERL_HOME_DIR}/.cpan/CPAN"
  chown -R shell:shell "${PERL_HOME_DIR}"
fi

# ---------------------------------------------------------------------
# configure CPAN
# 
if [ ! -r "${CPAN_CONFIG_FILE_SOURCE}" ] ; then
  LogMsg "No config file for CPAN (${CPAN_CONFIG_FILE_FILENAME}) found in this Magisk Module"
else
  if [ -r "${CPAN_CONFIG_FILE}" ] ; then
    LogMsg "The config file for CPAN \"${CPAN_CONFIG_FILE}\" already exists"
    diff "${CPAN_CONFIG_FILE}" "${CPAN_CONFIG_FILE_SOURCE}" 2>/dev/null
    if [ $? -eq 0 ] ; then
      LogMsg "The existing config file for CPAN is already okay" 
      INSTALL_CPAN_CONFIG_FILE=${__FALSE}
    else
      LogMsg "Renaming the file \"${CPAN_CONFIG_FILE}\" to \"${CPAN_CONFIG_FILE}.$$.bak\" ..."
      mv "${CPAN_CONFIG_FILE}" "${CPAN_CONFIG_FILE}.$$.bak" || INSTALL_CPAN_CONFIG_FILE=${__FALSE}
    fi
  fi

  if [ ${INSTALL_CPAN_CONFIG_FILE} = ${__TRUE} ] ; then
    LogMsg "Installing the CPAN config file \"${CPAN_CONFIG_FILE}\" ..."
    cp "${CPAN_CONFIG_FILE_SOURCE}" "${CPAN_CONFIG_FILE}"
    chown -R shell:shell "${CPAN_CONFIG_FILE}"
  fi
fi

# ---------------------------------------------------------------------


LogMsg "Correcting the permissions for the new files ..."

cd "${MODPATH}"

# first the directories with data and config files 
# 
for i in etc include usr ; do

  [[ $i != /\* ]] && CUR_ENTRY="${MODPATH}/system/$i" || CUR_ENTRY="$i"

  [ ! -d "${CUR_ENTRY}" ] && continue

  LogMsg "Processing the files in the directory ${CUR_ENTRY} ..."
  set_perm_recursive "${CUR_ENTRY}" 0 0 0755 0644 u:object_r:system_file:s0
  chcon -R -h u:object_r:system_file:s0 "${CUR_ENTRY}"
done


# now the directories with binaries
# 
for i in bin lib lib64 usr/bin usr/clang${CLANG_VERSION}/bin usr/clang${CLANG_VERSION}/libexec ; do

  [[ $i != /\* ]] && CUR_ENTRY="${MODPATH}/system/$i" || CUR_ENTRY="$i"
	
  [ ! -d "${CUR_ENTRY}" ] && continue

  LogMsg "Processing the files in the directory ${CUR_ENTRY} ..."

  set_perm_recursive "${CUR_ENTRY}" 0 0 0755 0755 u:object_r:system_file:s0

  chcon -R -h  u:object_r:system_file:s0 "${CUR_ENTRY}"
done

