#
# Customiztion script for the Magisk Module AddBin
#
# History
#   08.08.2024 /bs
#     initial release
#
# Notes:
#
# This Magisk Module contains Perl 5.38 for arm64 CPUs
#
# Documentation for creating Magisk Modules: https://topjohnwu.github.io/Magisk/guides.html
#
# Environment variables that can be used:
#
#    MAGISK_VER (string): the version string of current installed Magisk (e.g. v20.0)
#    MAGISK_VER_CODE (int): the version code of current installed Magisk (e.g. 20000)
#    BOOTMODE (bool): true if the module is being installed in the Magisk app
#    MODPATH (path): the path where your module files should be installed
#    TMPDIR (path): a place where you can temporarily store files
#    ZIPFILE (path): your module’s installation zip
#    ARCH (string): the CPU architecture of the device. Value is either arm, arm64, x86, or x64
#    IS64BIT (bool): true if $ARCH is either arm64 or x64
#    API (int): the API level (Android version) of the device (e.g. 21 for Android 5.0)
#

# -----------------------------------------------------------------------------

# define constants
#
__TRUE=0
__FALSE=1


# -----------------------------------------------------------------------------
# init global variables
#

MODULE_VERSION="$( grep "^version=" $MODPATH/module.prop  | cut -f2 -d "=" )"


# -----------------------------------------------------------------------------

function LogMsg {
  ui_print "$*"
}

function LogInfo {
  LogMsg "INFO: $*"
}

function LogWarning {
  LogMsg "WARNING: $*"
}

function LogError {
  LogMsg "ERROR: $*"
}


# -----------------------------------------------------------------------------

LogMsg "The current environment for this installation is"

LogMsg "The version of the installed Magisk is \"${MAGISK_VER}\" (${MAGISK_VER_CODE})"

LogInfo "BOOTMODE is \"${BOOTMODE}\" "
LogInfo "MODPATH is \"${MODPATH}\" "
LogInfo "TMPDIR is \"${TMPDIR}\" "
LogInfo "ZIPFILE is \"${ZIPFILE}\" "
LogInfo "ARCH is \"${ARCH}\" "
LogInfo "IS64BIT is \"${IS64BIT}\" "
LogInfo "API is \"${API}\" "

# -----------------------------------------------------------------------------

# example output for the variables:


#  The version of the installed Magisk is "25.0" (25000)
#  INFO: BOOTMODE is "true" 
#  INFO: MODPATH is "/data/adb/modules_update/PlayStore_for_MicroG" 
#  INFO: TMPDIR is "/dev/tmp" 
#  INFO: ZIPFILE is "/data/user/0/com.topjohnwu.magisk/cache/flash/install.zip" 
#  INFO: ARCH is "arm64" 
#  INFO: IS64BIT is "true" 
#  INFO: API is "32"


# -----------------------------------------------------------------------------

LogMsg "Installing the Magisk Module for the additional binaries version \"${MODULE_VERSION}\" ..."

# LogMsg "Checking the OS configuration ..."

ERRORS_FOUND=${__FALSE}

MACHINE_TYPE="$( uname -m )"

PERL_HOME_DIR="/data/local/tmp/perl_home"

PERL_SOURCE_DIR="${MODPATH}/perl538"
PERL_TARGET_DIR="/data/local/tmp/perl538"

# check the current CPU
#
LogMsg "Checking the type of the CPU used in this device ...."
LogMsg "The CPU in this device is a ${ARCH} CPU"
LogMsg "The machine type reported by \"uname -m\" is \"${MACHINE_TYPE}\" "

if [ "${ARCH}"x != "arm64"x -a  "${ARCH}"x != "arm"x ] ; then
  abort "This Magisk module is for arm and x86 CPUs only"
fi

if [ -d "${PERL_TARGET_DIR}" ] ; then
  BACKUP_EXTENSION="$$_$( date +%Y_%m-%d )"
  LogMsg "Renaming the existing directory \"${PERL_TARGET_DIR}\" to \"${PERL_TARGET_DIR}.${BACKUP_EXTENSION}\" ..."
  mv "${PERL_TARGET_DIR}" "${PERL_TARGET_DIR}.${BACKUP_EXTENSION}"
fi
 
LogMsg "Copying the Perl files from \"${PERL_SOURCE_DIR}\" to \"${PERL_TARGET_DIR}\" ..."

mkdir -p "${PERL_TARGET_DIR}" && cp -r "${PERL_SOURCE_DIR}"/* "${PERL_TARGET_DIR}"

LogMsg "Correcting the permissions for the Perl files ..."

set_perm_recursive "${PERL_TARGET_DIR}/bin" 0 0 0755 0755 u:object_r:system_file:s0
set_perm_recursive "${PERL_TARGET_DIR}/lib" 0 0 0755 0444 u:object_r:system_file:s0

# change the SELinux context for symbolic links also
#
chcon -R -h  u:object_r:system_file:s0 "${PERL_TARGET_DIR}"

find "${PERL_TARGET_DIR}/lib" -name "*.so" -exec chmod 755 {} \;
find "${PERL_TARGET_DIR}/lib" -name "*.pod" -exec chmod 644 {} \;

# ---------------------------------------------------------------------
#
if [ -d ${PERL_HOME_DIR} ] ; then
  LogMsg "The Perl home directory \"${PERL_HOME_DIR}\" already exists"
else
  LogMsg "Creating the Perl home directory \"${PERL_HOME_DIR}\" ..."
  mkdir -p "${PERL_HOME_DIR}" 
  mkdir -p "${PERL_HOME_DIR}/perl5"
  chown -R shell:shell "${PERL_HOME_DIR}"
fi

# ---------------------------------------------------------------------
#  
LogMsg "Creating the script to set the environment for Perl  \"/system/bin/perl_env\" ..."
cat >${MODPATH}/system/bin/perl_env <<EOT

export PATH=${PERL_TARGET_DIR}:\${PATH}

export LD_LIBRARY_PATH=${PERL_TARGET_DIR}/lib64:${LD_LIBRARY_PATH}

export HOME="${PERL_HOME_DIR}"

export PERL5LIB="${PERL_HOME_DIR}/perl5"
EOT

LogMsg "Creating the wrapper script \"/system/bin/perl_wrapper\" ..."
cat >${MODPATH}/system/bin/perl_wrapper <<EOT
#!/system/bin/sh

. /system/bin/perl_env

if [ \${0##*/} = perl_wrapper ] ; then
  echo "ERROR: Do not call this script - this is only a wrapper used for the Perl executables"
  exit 254
fi


eval 'exec  ${PERL_TARGET_DIR}/bin/\${0##*/}  \${1+"\$@"}'

EOT

for i in perl cpan perlbug perldoc ; do
  LogMsg "Creating the linkw \"/system/bin/$i\" ..."
  ln -s ./perl_wrapper "${MODPATH}/system/bin/$i"
done

set_perm_recursive "${MODPATH}/system/bin" 0 0 0755 0755 u:object_r:system_file:s0
chcon -R -h  u:object_r:system_file:s0 "${MODPATH}/system/bin"





