#
# Customization script for the Magisk Module for btop
#
# History
#   15.03.2023 /bs
#     initial release
#   04.05.2023 /bs
#     corrected some spelling errors in the documentation 
#   24.02.2024 /bs
#     updated the binaries to version 1.3.2
#   13.04.2024 /bs
#     removed the unneccesary directory system/xbin from the module
#
# Notes:
#
# This Magisk Module contains btop binaries for x86-64 and arm-64 CPUs and other CPUs
#
# The binaries for btop are from https://github.com/aristocratos/btop/releases
#
# Documentation for creating Magisk Modules: https://topjohnwu.github.io/Magisk/guides.html
#
# Environment variables that can be used:
#
#    MAGISK_VER (string): the version string of current installed Magisk (e.g. v20.0)
#    MAGISK_VER_CODE (int): the version code of current installed Magisk (e.g. 20000)
#    BOOTMODE (bool): true if the module is being installed in the Magisk app
#    MODPATH (path): the path where your module files should be installed
#    TMPDIR (path): a place where you can temporarily store files
#    ZIPFILE (path): your module’s installation zip
#    ARCH (string): the CPU architecture of the device. Value is either arm, arm64, x86, or x64
#    IS64BIT (bool): true if $ARCH is either arm64 or x64
#    API (int): the API level (Android version) of the device (e.g. 21 for Android 5.0)
#

# -----------------------------------------------------------------------------

# define constants
#
__TRUE=0
__FALSE=1


# -----------------------------------------------------------------------------
# init global variables
#

MODULE_VERSION="1.3.2.1"


FORCE_SEMAPHOR_FILE="/sdcard/Download/force_installation_of_btop"

BINARY=""

TARGET_BINARY="${MODPATH}/system/bin/btop.bin"
WRAPPER_SCRIPT="${MODPATH}/system/bin/btop"

# -----------------------------------------------------------------------------

function LogMsg {
  ui_print "$*"
}

function LogInfo {
  LogMsg "INFO: $*"
}

function LogWarning {
  LogMsg "WARNING: $*"
}

function LogError {
  LogMsg "ERROR: $*"
}

# returns: ${__TRUE}  - semaphor exists
#          ${__FALSE} - semaphor does NOT exist
#
function check_for_force_semaphor {
  LOCAL_THISRC=${__FALSE}

  if [ "${FORCE_SEMAPHOR_FILE}"x != ""x  ] ; then
    if [ -r "${FORCE_SEMAPHOR_FILE}" ] ; then
       LogInfo "The file \"${FORCE_SEMAPHOR_FILE}\" exists -- the installation will continue"
       LOCAL_THISRC=${__TRUE}
    else
      LogMsg "Create the file \"${FORCE_SEMAPHOR_FILE}\" before installing this Magisk Module to force the installation"
    fi
  fi

 return ${LOCAL_THISRC}
}

# -----------------------------------------------------------------------------

LogMsg "The current environment for this installation is"

LogMsg "The version of the installed Magisk is \"${MAGISK_VER}\" (${MAGISK_VER_CODE})"

LogInfo "BOOTMODE is \"${BOOTMODE}\" "
LogInfo "MODPATH is \"${MODPATH}\" "
LogInfo "TMPDIR is \"${TMPDIR}\" "
LogInfo "ZIPFILE is \"${ZIPFILE}\" "
LogInfo "ARCH is \"${ARCH}\" "
LogInfo "IS64BIT is \"${IS64BIT}\" "
LogInfo "API is \"${API}\" "

# -----------------------------------------------------------------------------

# example output for the variables:


#  The version of the installed Magisk is "25.0" (25000)
#  INFO: BOOTMODE is "true" 
#  INFO: MODPATH is "/data/adb/modules_update/PlayStore_for_MicroG" 
#  INFO: TMPDIR is "/dev/tmp" 
#  INFO: ZIPFILE is "/data/user/0/com.topjohnwu.magisk/cache/flash/install.zip" 
#  INFO: ARCH is "arm64" 
#  INFO: IS64BIT is "true" 
#  INFO: API is "32"


# -----------------------------------------------------------------------------

LogMsg "Installing the Magisk Module for btop \"${MODULE_VERSION}\" ..."

# LogMsg "Checking the OS configuration ..."

ERRORS_FOUND=${__FALSE}

MACHINE_TYPE="$( uname -m )"

# check the current CPU
#
LogMsg "Checking the type of the CPU used in this device ...."
LogMsg "The CPU in this device is a ${ARCH} CPU"
LogMsg "The machine type reported by \"uname -m\" is \"${MACHINE_TYPE}\" "

if [ ${ERRORS_FOUND} = ${__TRUE} ] ; then
  check_for_force_semaphor || abort "ERRROR: One or more errors in the prerequisite checks found"
fi

if [ -r $MODPATH/bin/btop_${ARCH} ] ; then
  BINARY="$MODPATH/bin/btop_${ARCH}"
elif [ -r $MODPATH/bin/btop_${MACHINE_TYPE} ] ; then
  BINARY="$MODPATH/bin/btop_${MACHINE_TYPE}"
else
  abort "[No binary for the current CPU found]: Install will now exit"
fi

if [ "${BINARY}"x != ""x ] ; then
  cp "${BINARY}" "${TARGET_BINARY}"
  if [ $? -eq 0 ]; then 
	ui_print "[File copied]: ${BINARY} to ${TARGET_BINARY}" 
	set_perm "${TARGET_BINARY}" 0 0 555
	set_perm "${WRAPPER_SCRIPT}" 0 0 555

  else
	ui_print "[File NOT copied]: ${BINARY} to ${TARGET_BINARY}" 
	abort "[File NOT copied]: Install will now exit" 
  fi
fi


