#!/system/bin/sh
#h#
#h# uninstall_playstore_update.sh #VERSION# - uninstall a PlayStore update
#h#
#h# Usage: uninstall_playstore_update.sh  [-h|--help] [-H] [-d|--dryrun] [-x|--doit] [-f|--force] [-c|--clean] [-s|--stop] [-S|--Stop] [-r|--reboot] [-v|--verbose] [--action] [-V|--version] [var=value]
#h#
#H# Known parameter:
#H# 
#H# -h               print the short usage help
#H# -H               print the detailed usage help
#H# -d               run the script in dry-run mode
#H# -x               delete the PlayStore update if the PlayStore is currently not running
#H# -f               delete the PlayStore update even if the PlayStore is currently running
#H# -c               cleanup the update and cache directories for the PlayStore
#H# -s               force stop the PlayStore
#H# -S               force stop the PlayStore and exit the script
#H# -r               reboot the phone after deleting the PlayStore update
#H# -v               print more messages
#H# -V               print the script version and exit
#H# --action         the script is running from within the action.sh script from Magisk
#H# var=value        set the variable "var" to the value "value"
#H#
#H# Return codes:
#H#
#H#   0 - no PlayStore update found
#H#   1 - PlayStore update found but not deleted
#H#   2 - PlayStore update deleted; reboot required
#H#   3 - PlayStore update found and the PlayStore is running
#H#   4 - no installed PlayStore found
#H#   5 - cache directories deleted
#H#
#H# Set the variable PREFIX to "echo" or something similar to run the script in dry-run mode
#H# Set the environment variable TRACE to any value to run the script with "set -x"
#H#
#
#
# Lines beginning with #h# are printed if the script is executed with the parameter "-h", "--help" or "-H"
#
# Lines beginning with #H# are printed if the script is executed with the parameter "-H"
#
# Author
#   Bernd Schemmer (bernd dot schemmer at gmx dot de)
#
# History
#   06.09.2025 /bs v1.0.0
#     initial release
#   07.09.2025 /bs v1.1.0
#     the script now prints also the version of the installed PlayStore
#     added the parameter "-c" to clean the cache directories only
#     the script now prints the version from the PlayStore apks used if the binary aapt2 is available via PATH
#   10.09.2025 /bs v1.2.0
#     the script now restarts itself as user root if necessary and possible
#   16.09.2025 /bs 1.3.0
#     added the parameter "-s|--stop" and "-S" to stop the PlayStore
#   22.09.2025 /bs 1.4.0
#     added support for running the script in the action.sh script from Magisk
#     added code to detect incomplete removals of the PlayStore Update
#     the script now reads the VolumeUp/VolumeDown key before doing anything if the variable IN_ACTION_SCRIPT is ${__TRUE}
#     added the parameter "--action"
#   01.10.2025 /bs 1.5.0
#     the script now prints a warning if the FakeStore apk file is still active
#
# ----------------------------------------------------------------------
# define constants
#

__TRUE=0
__FALSE=1

# ---------------------------------------------------------------------
# global variables
#

PLAYSTORE_APP="com.android.vending"

# Script return code
#
THISRC=0

SCRIPT_PARAMETER="$*"

# ----------------------------------------------------------------------
# enable tracing if requested
#
if [ "${TRACE}"x != ""x ] ; then
  set -x
elif [[ $- == *x* ]] ; then
#
# tracing is already enabled 
#
  TRACE=${__TRUE}
fi

# ----------------------------------------------------------------------
# enable verbose mode if requested
#

if [[ " $* " == *\ -v\ * || " $* " == *\ --verbose\ *  ]] ; then
  VERBOSE=${__TRUE}
fi

# ----------------------------------------------------------------------
# install a trap handler for house keeping
#
trap "cleanup"  0

# ----------------------------------------------------------------------
# read the script version from the source code
#
SCRIPT_VERSION="$( grep  "^#" $0 | grep "/bs v"  | tail -1 | sed "s#.*v#v#g" )"

# ---------------------------------------------------------------------
# aliase
#
alias LogInfoVar='f() { [[ ${__FUNCTION} = "" ]] && __FUNCTION=main ; [[ ${VERBOSE} != 0 ]] && return; varname="$1"; eval "echo \"INFO: in $__FUNCTION:  $varname ist \${$varname}\" >&2"; unset -f f; } ;  f'

# ---------------------------------------------------------------------
# functions

# ---------------------------------------------------------------------
# LogMsg - write a message to STDOUT
#
# Usage: LogMsg [message]
#
function LogMsg {
  typeset __FUNCTION="LogMsg"
  
  typeset THISMSG="$@"

  THISMSG="$( echo "${THISMSG}" )"
  
  echo "${THISMSG}"

  return ${__TRUE}
}

# ---------------------------------------------------------------------
# LogInfo - write a message to STDERR if VERBOSE is ${__TRUE}
#
# Usage: LogInfo [message]
#
# The function  returns ${__TRUE} if the message was written and
# ${__FALSE} if the message was not written
#
function LogInfo { 
  typeset __FUNCTION="LogInfo"

  [[ ${VERBOSE} == ${__TRUE} ]] && LogMsg "INFO: $@" >&2 || return ${__FALSE}
}

# ---------------------------------------------------------------------
# LogWarning - write a warning message to STDERR
#
# Usage: LogWarning [message]
#
function LogWarning {
  typeset __FUNCTION="LogWarning"

  LogMsg "WARNING: $@" >&2
}


# ---------------------------------------------------------------------
# LogError - write an error message to STDERR
#
# Usage: LogError [message]
#
function LogError {
  typeset __FUNCTION="LogError"

  LogMsg "ERROR: $@" >&2
}


# ---------------------------------------------------------------------
# die - end the program
#
# Usage:
#  die [returncode] [message]
# 
# returns:
#   n/a
#
function die {
  typeset __FUNCTION="die"

  typeset THISRC="$1"

  [ "${THISRC}"x = ""x ] && THISRC=0

  [ $# -gt 0 ] && shift

  typeset THISMSG="$*"
  
  if [ ${THISRC} -le 4 ] ; then
    [ "${THISMSG}"x != ""x ] && LogMsg "${THISMSG} (RC=${THISRC})"
  else
    LogError "${THISMSG} (RC=${THISRC})"
  fi

  exit ${THISRC}
}

# ----------------------------------------------------------------------
# isNumber
#
# check if a value is an integer
#
# usage: isNumber testValue
#
# returns: ${__TRUE} - testValue is a number else not
#
function isNumber {

# this code does not work in the sh in Android
#  [[ $1 == +([0-9]) ]] && THISRC=${__TRUE} || THISRC=${__FALSE}

# old code:
  if [ "$1"x != ""x ] ; then
    TESTVAR="$(echo "$1" | sed 's/[0-9]*//g' )"
    [ "${TESTVAR}"x = ""x ] && return ${__TRUE} || return ${__FALSE}
  fi
  
  return ${__FALSE}
}

# ---------------------------------------------------------------------
# ReadVolumeKeys - read the status of the volume keys
#
# Usage: ReadVolumeKeys [timeout] 
#
# Parameter:
#        timeout - timeout in seconds to wait; the default value is 10 seconds
#
# returns:
#        0 - no volume key pressed 
#        1 - volume up pressed
#        2 - volume down pressed
#
function ReadVolumeKeys {
  typeset __FUNCTION="ReadVolumeKeys"

  typeset THISRC=0
   
  typeset DEFAULT_TIMEOUT=10
  
  typeset TIMEOUT="${DEFAULT_TIMEOUT}"
  typeset MESSAGE=""
  
  typeset dev=""
  typeset type=""
  typeset code=""
  typeset value=""
  
  typeset RESULT=""

  if [ $# -eq 1 ] ; then
    TIMEOUT=$1
    shift
  fi

# the variables dev, type, code, and value can not be used here because the "while" statement is running a separate session
#
  RESULT=$( timeout ${TIMEOUT} getevent -ql | while read dev type code value; do
    if [ "$code" = "KEY_VOLUMEUP" ] && [ "$value" = "DOWN" ]; then
      echo 1
      break
    elif [ "$code" = "KEY_VOLUMEDOWN" ] && [ "$value" = "DOWN" ]; then
      echo 2
      break
    fi
  done )
   
  [ "${RESULT}"x != ""x ] && THISRC="${RESULT}"
  
  return ${THISRC}
}

# ---------------------------------------------------------------------
# cleanup - house keeping at script end
#
# Usage:
#  cleanup
# 
# returns:
#   this function is used as trap handler to cleanup the environment
#
function cleanup {

  LogInfo "cleanup from $0 is running ..."

#
# remove the trap handler
#
  trap ""  0

  if [ "${PREFIX}"x != ""x ] ; then
    LogMsg ""
    LogMsg "*** The variable PREFIX is defined (PREFIX=\"${PREFIX}\") -- the script was executed in dry-run mode"
  fi

  LogMsg ""
  
# cleanup the environment 

}


# ----------------------------------------------------------------------
# install the trap handler
#
trap "cleanup" 0

# ----------------------------------------------------------------------
#
if [ "${PREFIX}"x != ""x ] ; then
# 
# check mksh (in some mksh versions PREFIX is used for a directory name)
#
  if [ -d "${PREFIX}" ] ; then
    LogWarning "The variable PREFIX contains a directory name: \"${PREFIX}\" -- disabling dry-run mode now (use the parameter \"-d\" to enable dry-run mode"
    PREFIX=""
  fi
fi

# ---------------------------------------------------------------------

# ---------------------------------------------------------------------
# main function


# ---------------------------------------------------------------------
# process the script parameter
#

LogInfo "Processing the parameter ..."

LogInfo "The parameter for the script are "  && \
  LogMsg "$*"


FORCE=${__FALSE}

REBOOT_PHONE=${__FALSE}

DOIT=${__FALSE}

PRINT_USAGE_HELP=${__FALSE}

PRINT_DETAILED_USAGE_HELP=${__FALSE}

CLEAN_DIRS=${__FALSE}

STOP_THE_PLAYSTORE=${__FALSE}
STOP_ONLY=${__FALSE}

while [ $# -ne 0 ] ; do
  CUR_PARAMETER="$1"
  shift

  LogInfo "Processing the parameter \"${CUR_PARAMETER}\" ..."

  case  ${CUR_PARAMETER} in

    -h | --help )
      PRINT_USAGE_HELP=${__TRUE}
      ;;

    -H )
      PRINT_USAGE_HELP=${__TRUE}
      PRINT_DETAILED_USAGE_HELP=${__TRUE}
      ;;
  
    -d | --dryrun )
      PREFIX="echo"
      ;;

    -c | --clean )
      CLEAN_DIRS=${__TRUE}
      ;;

    -s | --stop )
      STOP_THE_PLAYSTORE=${__TRUE}
      ;;

    -S | --Stop )
      STOP_THE_PLAYSTORE=${__TRUE}
      STOP_ONLY=${__TRUE}
      ;;              
      
    -f | --force )
      FORCE=${__TRUE}
      DOIT=${__TRUE}
      ;;

    -r | --reboot )
      REBOOT_PHONE=${__TRUE}
      ;;

    -x | --doit )
      DOIT=${__TRUE}
      ;;

    *=* )
      LogInfo "Executing now \"${CUR_PARAMETER}\" ..."
      eval ${CUR_PARAMETER}
      if [ $? -ne 0 ] ; then
        die 70 "Error executing \"${CUR_PARAMETER}\" "
      fi
      ;;

   -v | --verbose )
      VERBOSE=${__TRUE}
      ;;
       
   -V | --version )
      echo "${SCRIPT_VERSION}"
      die 0
      ;;

    --action )
      IN_ACTION_SCRIPT=${__TRUE}
      ;;


    --* | -* )
      die 17 "Unknown option found in the parameter: \"${CUR_PARAMETER}\")"
      ;;

    *  )
      die 19 "Unknown parameter found: \"${CUR_PARAMETER}\")"
      ;;

  esac
done

# ---------------------------------------------------------------------

if [ "${PREFIX}"x != ""x ] ; then
  LogMsg ""
  LogMsg "*** The script is running in dry-run mode: PREFIX is \"${PREFIX}\" "
  LogMsg ""
fi

# ---------------------------------------------------------------------

if [ ${PRINT_USAGE_HELP} = ${__TRUE} ] ; then
  grep "^#h#" $0 | cut -c4- | sed \
        -e "s/#VERSION#/${SCRIPT_VERSION}/g" 
  
  if [ ${PRINT_DETAILED_USAGE_HELP} = ${__TRUE} ] ;then
    grep "^#H#" $0 | cut -c4- 
  else
    echo " Use the parameter \"-H\" to print the detailed usage help"
  fi
              
  die 0
fi

# ---------------------------------------------------------------------
# check pre-requisites for the script
#
THIS_USER=$( id -un )

if [ "${THIS_USER}"x != "root"x ] ; then

# check for root access via su
#
  su - -c id 2>/dev/null >/dev/null
  if [ $? -eq 0 ] ; then
   echo "Restarting the script as user \"root\" ..."
#
# root access via su is workgin
#
    exec su - -c $0 ${SCRIPT_PARAMETER}
  
    die 200 "Restarting the script as user \"root\" via \"su - -c $0 $*\" failed"
  else
    die 100 "This script needs root access rights (the current user is \"${THIS_USER}\")"
  fi
fi

AAPT="$( which aapt2 )"

# print the messages with parameter hints?
#
NO_PARAMETER_MSG=${__FALSE}

# default: do not uninstall the update if the PlayStore is running
#
  UNINSTALL_PLAYSTORE_UPDATE=${__FALSE}
  
  DUMPSYS_CONTENTS="$( dumpsys package "${PLAYSTORE_APP}" )"
  
  INITIAL_PLAYSTORE_APK="$( echo "${DUMPSYS_CONTENTS}" | grep codePath= | tail -1 | cut -f2 -d "=" )/Phonesky.apk"
  
# retrieve the version of the installed PlayStore
#
  PLAYSTORE_ACTIVE_VERSION="$( echo "${DUMPSYS_CONTENTS}" | grep versionName | cut -f2 -d=  | head -1 )"

  UPDATE_APK_VERSION=""
  INITIAL_APK_VERSION=""

  LogMsg "The active PlayStore version is ${PLAYSTORE_ACTIVE_VERSION}"
  
# check if the PlayStore is running
#  
  ps -ef | grep -v grep | grep "${PLAYSTORE_APP}$" >/dev/null
  if [ $? -eq 0 ] ; then
    LogMsg "The PlayStore is running"
    PLAYSTORE_IS_RUNNING=${__TRUE}
  else
    PLAYSTORE_IS_RUNNING=${__FALSE}    
  fi

# check if the FakeStore is active
#
  FAKE_STORE_APK="$( ls -ld /*/priv-app/FakeStore/FakeStore.apk 2>/dev/null )"
  if [ "${FAKE_STORE_APK}"x != ""x ] ; then
    LogMsg "
 **********************************************************************
 ERROR: The FakeStore is still visible:
 
 ${FAKE_STORE_APK}
 
 The PlayStore does NOT work in this configuration!
 
 Please uninstall the Magisk Module with the PlayStore, reboot the phone
 and reinstall the Magisk Module with the PlayStore
 **********************************************************************
  
"
  fi
       
# stop the PlayStore
#
  if [ ${STOP_THE_PLAYSTORE} = ${__TRUE} ] ; then
    LogMsg "Doing now a force stop for the PlayStore ..."
    am force-stop "${PLAYSTORE_APP}"
    
    [ ${STOP_ONLY} = ${__TRUE} ] && die 0
  fi
  
# Find the installed PlayStore apk
# 
  LogMsg "Retrieving the current state of the PlayStore ..."
  CURRENT_PKG_DIR=$( pm path "${PLAYSTORE_APP}" | sed 's/package://;s/base.apk//' | head -1 )

  if [ ! -r "${INITIAL_PLAYSTORE_APK}" ] ; then
     LogMsg "The apk file \"${INITIAL_PLAYSTORE_APK}\" does not exist"
  elif [ "${AAPT}"x != ""x ] ; then
    set -- $( ${AAPT} dump badging "${INITIAL_PLAYSTORE_APK}" | grep versionName ) 
    INITIAL_APK_VERSION="${4#*\'}"
    if [ "${INITIAL_PLAYSTORE_APK}"x != ""x ] ; then
      LogMsg "The version of the initial installed PlayStore apk file \"${INITIAL_PLAYSTORE_APK}\" is ${INITIAL_APK_VERSION}" 
    fi
  fi
  
  if [ ${CLEAN_DIRS} = ${__TRUE} ] ; then
    LogMsg "Cleaning the update and cache directories for the PlayStore ..."

    LogMsg "Deleting the files in in the cache directories ..."
    set -x
    ${PREFIX} rm -rf /data/system/package_cache/*/*vending* 

    ${PREFIX} rm -rf  "/data/data/${PLAYSTORE_APP}"
    set +x
  elif [ -n "${CURRENT_PKG_DIR}" ] && [ -d "${CURRENT_PKG_DIR}" ]; then

    if [ -r "${CURRENT_PKG_DIR}/base.apk" ] ; then
      LogMsg "There is an update for the PlayStore installed in the the directory \"${CURRENT_PKG_DIR}\":"
      LogMsg
      LIST_OF_NEW_FILES="$( find ${CURRENT_PKG_DIR} )"
      LogMsg "$( ls -ldZ ${LIST_OF_NEW_FILES} )"
      LogMsg

      if [ "${AAPT}"x != ""x ] ; then
        set -- $( ${AAPT} dump badging "${CURRENT_PKG_DIR}/base.apk" | grep versionName ) 
        UPDATE_APK_VERSION="${4#*\'}"
        if [ "${INITIAL_PLAYSTORE_APK}"x != ""x ] ; then
          LogMsg "The version of the apk file with the PlayStore update is ${UPDATE_APK_VERSION}"    
        fi
      fi
    else
      if [ "${INITIAL_APK_VERSION}"x != "${UPDATE_APK_VERSION}"x ] ; then
        LogMsg 
        LogWarning "There is a PlayStore update installed but there are no files in the update directory   
  ${CURRENT_PKG_DIR}
anymore (most probably the reboot after deleting the PlayStore update is missing)"
  
        LogMsg "Please reboot the phone to fix the config"

        UNINSTALL_PLAYSTORE_UPDATE=${__FALSE}
        DOIT=${__FALSE}
        NO_PARAMETER_MSG=${__TRUE}


          if [ ${IN_ACTION_SCRIPT}x = ${__TRUE}x ] ; then    
            LogMsg "
*** Press \"Volume Down\" to reboot the phone now
       or \"Volume Up\" to cancel the reboot
       or wait 10 seconds to cancel the reboot
      
      "
            ReadVolumeKeys 10
      
            case $? in
      
               0 )
                 LogMsg "No Volume key pressed - reboot continues"
                 REBOOT_PHONE=${__FALSE}
                 ;;
                 
               1 )
                 LogMsg "VolumeUp key pressed - reboot canceled"
                 REBOOT_PHONE=${__FALSE}
                 ;;
                 
               2 )         
                 LogMsg "VolumeDown key pressed - rebooting the phone now"
                 ;;
                 
               * )
                 LogMsg "Unpexpected result while reading the VolumeKey - please reboot manually"
                 REBOOT_PHONE=${__FALSE}
                 ;;
            esac

            if [ ${REBOOT_PHONE} = ${__TRUE} ] ; then    
              LogMsg "Rebooting the phone to activate the original PlayStore version ..."
              ${PREFIX} reboot
            else
              IN_ACTION_SCRIPT=${__FALSE}

              LogMsg "Reboot canceled"
            fi
          fi
      fi
    fi
    
    if [ ${PLAYSTORE_IS_RUNNING} = ${__TRUE} ] ; then

      if [ ${FORCE} != ${__TRUE} ] ; then
        UNINSTALL_PLAYSTORE_UPDATE=${__FALSE}
        LogMsg "Nothing to do right now (use the parameter \"-f\"  or \"--force\" to uninstall the PlayStore update anyway "
        THISRC=3
      else
        LogMsg "Parameter \"-f\" found -- deleting the PlayStore update anyway"
        UNINSTALL_PLAYSTORE_UPDATE=${__TRUE}    
      fi
    else  
      LogMsg "The PlayStore is not running - looks like the PlayStore update is not working "
      UNINSTALL_PLAYSTORE_UPDATE=${__TRUE}        
    fi

    if [ ${IN_ACTION_SCRIPT}x = ${__TRUE}x ] ; then    
      LogMsg "
*** Press \"Volume Down\" to uninstall the PlayStore Update
    and reboot the phone
       or \"Volume Up\" to cancel 
    (or wait for 10 seconds to cancel the action)

"
  
      ReadVolumeKeys 10

      TEMPRC="$?"
      case ${TEMPRC} in

         0 )
           LogMsg "No Volume key pressed - doing nothing"
           UNINSTALL_PLAYSTORE_UPDATE=${__FALSE}
           ;;
           
         1 )
           LogMsg "VolumeUp key pressed - doing nothing"
           UNINSTALL_PLAYSTORE_UPDATE=${__FALSE}
           ;;
           
         2 )         
           LogMsg "VolumeDown key pressed"
           UNINSTALL_PLAYSTORE_UPDATE=${__TRUE}
           DOIT=${__TRUE}
           ;;
           
         * )
           LogMsg "Unexpected result \"${TEMPRC}\" when reading the VolumeKey - doing nothing"
           UNINSTALL_PLAYSTORE_UPDATE=${__FALSE}
           ;;
      esac
    fi 

    if [ ${UNINSTALL_PLAYSTORE_UPDATE} = ${__TRUE} ] ; then
      if [ ${DOIT} = ${__TRUE} ] ; then
      
        LogMsg "Deleting the PlayStore update ..."
        
        ${PREFIX} pm uninstall --user 0 "${PLAYSTORE_APP}" 

        ${PREFIX} pm clear "${PLAYSTORE_APP}"

        LogMsg "Deleting the files in \"${CURRENT_PKG_DIR}\" ..."
        
        ${PREFIX} rm -rf "${CURRENT_PKG_DIR}"/*

        LogMsg "Deleting the files in in the cache directories ..."

        ${PREFIX} rm -rf /data/system/package_cache/*/*vending* 

        ${PREFIX} rm -rf /data/system/package_cache/*/*FakeStore* 

        ${PREFIX} rm -rf /data/user/0/com.android.vending/

        ${PREFIX} rm -rf  "/data/data/${PLAYSTORE_APP}"
        
        LogMsg "Reinstalling the original version of the PlayStore ..."
        
        ${PREFIX} cmd package install-existing  "${PLAYSTORE_APP}"
        
        if [ ${REBOOT_PHONE} = ${__TRUE} ] ; then    

          if [ ${IN_ACTION_SCRIPT}x = ${__TRUE}x ] ; then    
            LogMsg "
The phone reboots in 10 seconds now ..
*** Press \"Volume Down\" to reboot the phone now
       or \"Volume Up\" to cancel the reboot
      
      "
            ReadVolumeKeys 10
            TEMPRC="$?"
            
            case ${TEMPRC} in
      
               0 )
                 LogMsg "No Volume key pressed - reboot continues"
                 ;;
                 
               1 )
                 LogMsg "VolumeUp key pressed - reboot canceled"
                 REBOOT_PHONE=${__FALSE}
                 ;;
                 
               2 )         
                 LogMsg "VolumeDown key pressed - rebooting the phone now"
                 ;;
                 
               * )
                 LogMsg "Unexpected result \"${TEMPRC}\" when reading the VolumeKey - doing nothing"
                 REBOOT_PHONE=${__FALSE}
                 ;;
            esac
          fi 

          if [ ${REBOOT_PHONE} = ${__TRUE} ] ; then    
            LogMsg "Rebooting the phone to activate the original PlayStore version ..."
            ${PREFIX} reboot
          else
            LogMsg "Reboot canceled"
          fi
        else
          LogMsg "Reboot the phone to activate the original PlayStore (use the parameter \"-r\" or \"--reboot\" to automatically reboot after deleting the PlayStore update"
        fi
  
        THISRC=2
      else
        [ ${NO_PARAMETER_MSG} != ${__TRUE} ] && \
          LogMsg "Use the parameter \"-x\" or \"--doit\" to delete the PlayStore update"  
        THISRC=1
      fi
    fi
  else
    [ ${NO_PARAMETER_MSG} != ${__TRUE} ] && \
      LogMsg "No PlayStore update found (The current path to the PlayStore is \"${CURRENT_PKG_DIR}\" )"

    THISRC=4
  fi

  return ${THISRC}
