#
# Customization script for the Magisk Module with Debug Tools 
#
# History
#   12.10.2024 /bs
#     initial release
#   18.10.2024 /bs
#     added code to add the directories in /data/local/tmp for tmux
#
# Notes:
#
# This Magisk Module contains files for arm64 CPUs
#
# Documentation for creating Magisk Modules: https://topjohnwu.github.io/Magisk/guides.html
#
# Environment variables that can be used:
#
#    MAGISK_VER (string): the version string of current installed Magisk (e.g. v20.0)
#    MAGISK_VER_CODE (int): the version code of current installed Magisk (e.g. 20000)
#    BOOTMODE (bool): true if the module is being installed in the Magisk app
#    MODPATH (path): the path where your module files should be installed
#    TMPDIR (path): a place where you can temporarily store files
#    ZIPFILE (path): your module’s installation zip
#    ARCH (string): the CPU architecture of the device. Value is either arm, arm64, x86, or x64
#    IS64BIT (bool): true if $ARCH is either arm64 or x64
#    API (int): the API level (Android version) of the device (e.g. 21 for Android 5.0)
#

# -----------------------------------------------------------------------------

# define constants
#
__TRUE=0
__FALSE=1

# -----------------------------------------------------------------------------
# init global variables
#

MODULE_VERSION="$( grep "^version=" $MODPATH/module.prop  | cut -f2 -d "=" )"

MODULE_NAME="$( grep "^name=" $MODPATH/module.prop  | cut -f2 -d "=" )"

MODULE_DESC="${MODULE_NAME} ${MODULE_VERSION}"

DATA_BASE_DIR="/data/local/tmp"

LOCAL_DATA_DIRS="
${DATA_BASE_DIR}/etc
${DATA_BASE_DIR}/var
${DATA_BASE_DIR}/var/tmp"

CUR_DATA_DIR=""

# -----------------------------------------------------------------------------
#
if [ 0 = 1 -o -r /data/local/tmp/debug ] ; then
  LOGFILE="/data/local/tmp/${MODULE_NAME}_customize.log"

  ui_print "Writing all messages to the log file ${LOGFILE}"
  exec 1>${LOGFILE} 2>&1
  set -x
fi

# -----------------------------------------------------------------------------


# -----------------------------------------------------------------------------

function LogMsg {
  ui_print "$*"
}

function LogInfo {
  LogMsg "INFO: $*"
}

function LogWarning {
  LogMsg "WARNING: $*"
}

function LogError {
  LogMsg "ERROR: $*"
}


# -----------------------------------------------------------------------------

LogMsg "The current environment for this installation is"

LogMsg "The version of the installed Magisk is \"${MAGISK_VER}\" (${MAGISK_VER_CODE})"

LogInfo "BOOTMODE is \"${BOOTMODE}\" "
LogInfo "MODPATH is \"${MODPATH}\" "
LogInfo "TMPDIR is \"${TMPDIR}\" "
LogInfo "ZIPFILE is \"${ZIPFILE}\" "
LogInfo "ARCH is \"${ARCH}\" "
LogInfo "IS64BIT is \"${IS64BIT}\" "
LogInfo "API is \"${API}\" "

# -----------------------------------------------------------------------------

# example output for the variables:


#  The version of the installed Magisk is "25.0" (25000)
#  INFO: BOOTMODE is "true" 
#  INFO: MODPATH is "/data/adb/modules_update/PlayStore_for_MicroG" 
#  INFO: TMPDIR is "/dev/tmp" 
#  INFO: ZIPFILE is "/data/user/0/com.topjohnwu.magisk/cache/flash/install.zip" 
#  INFO: ARCH is "arm64" 
#  INFO: IS64BIT is "true" 
#  INFO: API is "32"


# -----------------------------------------------------------------------------

LogMsg "Installing the Magisk Module with ${MODULE_NAME} for Android \"${MODULE_VERSION}\" ..."

LogMsg "Checking the OS configuration ..."

ERRORS_FOUND=${__FALSE}

MACHINE_TYPE="$( uname -m )"

# check the current CPU
#
LogMsg "Checking the type of the CPU used in this device ...."

LogMsg "The CPU in this device is a ${ARCH} CPU"
LogMsg "The machine type reported by \"uname -m\" is \"${MACHINE_TYPE}\" "


if [ "${ARCH}"x != "arm64"x ] ; then
  abort "This Magisk module is for arm64 CPUs only"
fi

# ---------------------------------------------------------------------

LogMsg "Installing the Debug tools for Linux ..."


LogMsg "Creating the data directories ..."

for CUR_DIR in ${LOCAL_DATA_DIRS} ; do
  if [ -d "${CUR_DIR}" ] ; then
    LogMsg "The directory \"${CUR_DIR}\" already exists"
  else
    LogMsg "Creating the directory \"${CUR_DIR}\" ..."
    mkdir -p "${CUR_DIR}" && chown shell:shell "${CUR_DIR}"
    if [ $? -ne 0 ] ; then
      LogWarning "WARNING: Error creating the directory \"${CUR_DIR}\" "
    fi
  fi
done

echo 

if [ -r "${MODPATH}/example_tmux.conf" -a -d /data/local/tmp/etc ] ; then
  LogMsg "Creating the example tmux config file \"/data/local/tmp/etc/example_tmux.conf\" ..."
  cp "${MODPATH}/example_tmux.conf" /data/local/tmp/etc
fi

LogMsg "Correcting the permissions for the new files ..."

# first usr/* then bin, lib  etc !
# 
for i in etc include usr/include usr/lib  ; do

  [[ $i != /\* ]] && CUR_ENTRY="${MODPATH}/system/$i" || CUR_ENTRY="$i"

  [ ! -d "${CUR_ENTRY}" ] && continue

  LogMsg "Processing the files in the directory \"${CUR_ENTRY}\" ..."
  set_perm_recursive "${CUR_ENTRY}" 0 0 0755 0644 u:object_r:system_file:s0
  chcon -R -h u:object_r:system_file:s0 "${CUR_ENTRY}"
done

for i in bin lib lib64 usr/bin  ; do

  [[ $i != /\* ]] && CUR_ENTRY="${MODPATH}/system/$i" || CUR_ENTRY="$i"
	
  [ ! -d "${CUR_ENTRY}" ] && continue

  LogMsg "Processing the files in the directory \"${CUR_ENTRY}\" ..."

  set_perm_recursive "${CUR_ENTRY}" 0 0 0755 0755 u:object_r:system_file:s0

  chcon -R -h  u:object_r:system_file:s0 "${CUR_ENTRY}"
done

# create a symbolic link for nano if there is no nano in /system/bin
#

if [ -x ${MODPATH}/system/usr/bin/nano ] ; then
  echo "Creating the symbolic link /system/bin/nano -> /system/usr/bin/nano ..."
  [ ! -r /system/bin/nano ] && ln -s ../usr/bin/nano ${MODPATH}/system/bin/nano
fi

# the following code is currently not used
# 
if [ ! -z  ${CUR_DATA_DIR} ] ; then
  if [ -d "${CUR_DATA_DIR}" ] ; then
    LogMsg "The data directory \"${CUR_DATA_DIR}\" already exists"
  else
    LogMsg "Creating the data directory \"${CUR_DATA_DIR}\" ..."
    if [ ! -r "${CUR_DATA_TAR_ARCHIVE}" ] ; then
      LogError "The tar file with the contents for the data dir \"${CUR_DATA_TAR_ARCHIVE}\" does not exist"
    elif [ ! -d "${DATA_BASE_DIR}" ]; then
      LogError "The directory for the data files \"${DATA_BASE_DIR}\" does not exist"
    else
      cd "${DATA_BASE_DIR}" && tar -xzf "${CUR_DATA_TAR_ARCHIVE}" || \
        LogWarning "Error extracting the file \"${CUR_DATA_TAR_ARCHIVE}\" "
      set_perm_recursive "${DATA_BASE_DIR}/share" 2000 2000 0755 0644 u:object_r:system_file:s0
    fi
  fi
fi
